# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills, do NOT redefine primitives.

from video import init_video_writers, recording_step, recording_get_observation

# Module providing environment/object info
from object_positions import get_object_positions

def run_skeleton_task():
    '''Task runner that includes an exploration phase to diagnose the missing predicate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers, if needed
        init_video_writers(obs)

        # Hook up recording to task step and get_observation
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve all object positions in the environment
        positions = get_object_positions()

        # For exploration, print out available objects and their positions for debugging
        print("[Exploration] Detected objects and positions:", positions)

        # === Exploration Phase ===
        # The goal here is to determine which predicate is missing or unknown
        # For this, we attempt to use available primitives to gather information about the objects/drawers.
        # The domain and exploration knowledge suggest we should check predicates like
        # (identified ?obj), (temperature-known ?obj), (weight-known ?obj), (durability-known ?obj), (lock-known ?obj).
        # Based on missing plan action or planning issues (like "drawer-closed" or "lock-known"), we will attempt to gather information accordingly.

        # Available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # For exploration, use 'execute_go', 'execute_pick', 'execute_pull' as needed (no "identify", "temperature", etc. skill per skill_code list).

        # Try to "go" to each location and "pick"/"pull"/"push" various objects to see their effects and simulate knowledge gathering.
        # If objects or drawers exist according to the observation, we test interactions.

        # For demonstration, iterate over all objects that are on the floor (simulate "identify"/"pick"/"pull" etc.)
        objects = [name for name in positions.keys() if "drawer" not in name]
        drawers = [name for name in positions.keys() if "drawer" in name]
        locations = [positions[name] for name in positions.keys()]

        # Default: Try to move to every object and interact to see if any predicate triggers an error or logging diagnostic output.
        # Try to open drawers, pick objects, or "pull" handles to see which actions are blocked by unknown/missing predicates.

        # == Example Exploration Strategy ==
        for obj in objects:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Approaching object {obj} at {obj_pos}")
                # Move robot to object's position
                obs, reward, done = execute_go(env, task, target_pos=obj_pos)
                print(f"[Exploration] Arrived at {obj}.")
                # Try to pick up the object (if applicable)
                obs, reward, done = execute_pick(env, task, obj)
                print(f"[Exploration] Picked up {obj}.")
                # If successful, we can try placing it somewhere or check knowledge predicates
            except Exception as e:
                print(f"[Exploration] Could not pick {obj}: {e}")

        # Now, for drawers/handles: try to pull (open) all drawers
        for dr in drawers:
            try:
                drawer_pos = positions[dr]
                print(f"[Exploration] Approaching drawer {dr} at {drawer_pos}")
                obs, reward, done = execute_go(env, task, target_pos=drawer_pos)
                print(f"[Exploration] At drawer {dr}. Try to pull/open.")
                # Try to pull the handle or interact with the drawer
                # Need to first pick handle if modeled, then "pull"
                # For demonstration: try execute_pull with drawer and assume handle as parameter (optional)
                obs, reward, done = execute_pull(env, task, dr)
                print(f"[Exploration] Pulled (opened) {dr}.")
            except Exception as e:
                print(f"[Exploration] Could not pull {dr}: {e}")
        
        # Also, attempt to push drawers if they are open (simulate close)
        for dr in drawers:
            try:
                drawer_pos = positions[dr]
                print(f"[Exploration] Approaching drawer {dr} for push/close at {drawer_pos}")
                obs, reward, done = execute_go(env, task, target_pos=drawer_pos)
                obs, reward, done = execute_push(env, task, dr)
                print(f"[Exploration] Pushed (closed) {dr}.")
            except Exception as e:
                print(f"[Exploration] Could not push {dr}: {e}")

        # Optional: try sweeping or rotating objects (if such skills are meaningful in your domain)
        for obj in objects:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Sweeping object {obj}")
                obs, reward, done = execute_sweep(env, task, obj)
                print(f"[Exploration] Swept {obj}.")
            except Exception as e:
                print(f"[Exploration] Could not sweep {obj}: {e}")

        print("[Exploration] Finished exploration phase.")

        # === Place your actual task plan logic below this point ===
        # After the exploration phase, you would normally execute the oracle plan to achieve the desired goal.
        # For the moment, this code is focused on the exploration & missing predicate detection, as requested by feedback.

        # NOTE:
        # If a particular skill fails and logs a predicate or precondition missing (e.g., "drawer-closed", "drawer-unlocked" etc.),
        # this suggests which predicates are not handled or known by the planner, helping you patch your planning domain.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
