# run_skeleton_task.py (Exploration-enabled Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # You don't need to redefine primitives like move, pick, place

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation with exploration phase.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ... (dictionary format)
        positions = get_object_positions()

        # === EXPLORATION PHASE to detect missing predicates ===
        # This phase aims to explore the state of objects and locations to guess missing predicates
        object_names = list(positions.keys())
        explored_predicates = {}

        # Try to 'identify' each object by moving to its location
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Moving to object {obj} at position {obj_pos}")
                # Try to move to the object's position (simulate 'robot-at')
                obs, reward, done = execute_go(
                    env,
                    task,
                    target_pos=obj_pos,
                    max_steps=100,
                    threshold=0.05,
                    approach_axis='z'
                )
                if done:
                    print(f"[Exploration] Arrived at {obj}. Attempting to pick...")
                    # Try to pick the object (simulate 'execute_pick_weight'/'execute_pick_durability')
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=obj,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Picked {obj}.")
                        explored_predicates[obj] = 'pickable/on-floor/identified'
                        # Place back if needed
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj_name=obj,
                            target_pos=obj_pos,
                            drawer_name=None,
                            approach_distance=0.10,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                    else:
                        print(f"[Exploration] Could NOT pick {obj}.")
                else:
                    print(f"[Exploration] Could NOT approach {obj}.")
            except Exception as ex:
                print(f"[Exploration] Exception during exploring {obj}: {ex}")

        # Try to pull or manipulate handles (simulate missing 'lock-known' predicate detection)
        # We assume a naming or type convention for drawers and handles
        drawer_names = [name for name in object_names if 'drawer' in name]
        handle_names = [name for name in object_names if 'handle' in name or 'grip' in name or 'knob' in name]
        for drawer in drawer_names:
            # Try finding the handle of the drawer
            related_handles = [h for h in handle_names if drawer in h]
            for handle in related_handles:
                try:
                    handle_pos = positions[handle]
                    print(f"[Exploration] Attempting to pick handle {handle} for drawer {drawer}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=handle,
                        target_pos=handle_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Picked handle {handle}")
                        # Now try pull to check for lock
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            drawer_name=drawer,
                            handle_name=handle,
                            target_pos=handle_pos,
                            approach_distance=0.10,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        print(f"[Exploration] Pulled handle {handle} for drawer {drawer}")
                        explored_predicates[(drawer, handle)] = 'pull/lock-known-possible'
                    else:
                        print(f"[Exploration] Could not pick handle {handle}")
                except Exception as ex:
                    print(f"[Exploration] Exception at drawer {drawer} handle {handle}: {ex}")

        # === Report results of exploration (for predicate guessing) ===
        print("[Exploration] Results:")
        for k, v in explored_predicates.items():
            print(f"  {k}: {v}")

        # Based on exploration, you will know which predicate is missing (e.g. lock-known, on-floor, etc.)

        # === MAIN PLAN EXECUTION PHASE (after missing predicate detection) ===
        # Here you would insert your oracle plan, step by step, as a sequence of skill calls.
        # Example for a plan that moves, picks and places an object:

        for obj in object_names:
            try:
                obj_pos = positions[obj]
                # Move to the object's location
                print(f"[Task] Moving to {obj} at {obj_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    target_pos=obj_pos,
                    max_steps=100,
                    threshold=0.05,
                    approach_axis='z'
                )
                if done:
                    # Try to pick up the object
                    print(f"[Task] Attempting to pick {obj}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=obj,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Task] Picked up {obj}")
                        # Try to find a drawer to place the object
                        for drawer in drawer_names:
                            drawer_pos = positions.get(drawer, None)
                            if drawer_pos is not None:
                                # Move to the drawer position
                                print(f"[Task] Moving to drawer {drawer} at {drawer_pos}")
                                obs, reward, done = execute_go(
                                    env,
                                    task,
                                    target_pos=drawer_pos,
                                    max_steps=100,
                                    threshold=0.05,
                                    approach_axis='z'
                                )
                                # Place object into the drawer
                                print(f"[Task] Attempting to place {obj} into {drawer}")
                                obs, reward, done = execute_place(
                                    env,
                                    task,
                                    obj_name=obj,
                                    target_pos=drawer_pos,
                                    drawer_name=drawer,
                                    approach_distance=0.10,
                                    max_steps=100,
                                    threshold=0.01,
                                    timeout=10.0
                                )
                                if done:
                                    print(f"[Task] Placed {obj} into {drawer}")
                                    break
                    else:
                        print(f"[Task] Failed to pick up {obj}")
            except Exception as ex:
                print(f"[Task] Exception handling {obj}: {ex}")

        # You can further add PUSH or GRIPPER skills if required by your plan/goals

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
