# run_skeleton_task.py (Completed for Predicate Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills; do not redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # For this exploration, we will enumerate positions and look for unexplained constraints
        positions = get_object_positions()

        print("[Exploration] Object positions detected:", positions)

        # --- Predicate Exploration Phase: attempt to discover missing predicates ---
        # Feedback: (robot-at drawer_top_place_left) suggests a predicate is missing for this location.
        # We will attempt to execute movements to all known locations and monitor any failures
        # caused by missing predicates, and document those points as part of the exploration.

        # We'll use only the available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push',
            'execute_pull', 'execute_sweep', 'execute_rotate',
            'execute_go', 'execute_gripper'
        ]

        # Gather all detected locations
        all_locations = []
        for obj_name, pos_tuple in positions.items():
            # Heuristic: If obj_name contains 'place' or is a known location string, add it
            if 'place' in obj_name or 'drawer' in obj_name or 'table' in obj_name or '_left' in obj_name or '_right' in obj_name or '_center' in obj_name:
                all_locations.append(obj_name)
        # Remove duplicates
        all_locations = list(set(all_locations))

        # (In real code, ideally, location names would be available from observation; here we guess)
        # Ensure 'drawer_top_place_left' is included due to feedback
        if 'drawer_top_place_left' not in all_locations:
            all_locations.append('drawer_top_place_left')

        # For initial exploration, try moving (execute_go) to each location and log any issues
        robot_current_loc = None
        # Try to determine robot's current location from the observation or positions
        for obj in positions:
            if 'robot' in obj and obj != 'robot':
                robot_current_loc = positions[obj]
        # If not available, try defaulting to the first location
        if robot_current_loc is None and len(all_locations) > 0:
            robot_current_loc = all_locations[0]  # Placeholder

        print("[Exploration] All known locations to attempt:", all_locations)

        prev_location = None
        # Try sequential navigation
        # For exploration purposes, cycle through available locations
        for i, loc in enumerate(all_locations):
            try:
                if i == 0:
                    # Assume robot starts here for exploration purposes
                    prev_location = loc
                    continue
                print(f"[Exploration] Trying to move from {prev_location} to {loc} using execute_go")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=prev_location,
                    to_location=loc,
                    timeout=10.0
                )
                print(f"[Exploration] Success: robot-at {loc}")
                prev_location = loc
            except Exception as e:
                print(f"[Exploration] ERROR moving from {prev_location} to {loc}: {e}")
                print("-> Potential missing predicate or condition for this movement.")
                # Log feedback for missing predicate exploration
                # Optionally, break or continue (here, continue to try all)
                continue

        # You can append further exploration calls for other skills, e.g. execute_pick, execute_place, etc.,
        # on objects detected at these locations to further triangulate which predicates are missing

        print("[Exploration] Finished exploration phase.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
