# run_skeleton_task.py (Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with feedback-driven exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions = {'drawer2': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Find the missing predicate (drawer-open) for drawer2 ===
        #
        # According to the feedback: (drawer-open drawer2)
        # We'll use the available skills to perform exploratory actions to determine if 'drawer-open' (or other state) holds.
        #
        # Since 'execute_pull' can change the drawer from "drawer-closed" to "drawer-open", 
        # and 'execute_push' can close it, and 'execute_go' can move to the location,
        # we'll explore by using these actions, observe state changes, and confirm the relevant predicate.
        #
        # We'll assume objects and drawers are listed in the observation or object positions
        # and use names like 'drawer2' as needed.

        # Get relevant object names from positions or descriptions
        # Let's attempt to move to drawer2, try to open and close it, and check results

        robot_location = None
        drawer2_location = None
        handle_of_drawer2 = None

        # You may need to update these object names as per your environment setup
        # Try to infer (drawer2, handle_of_drawer2) and their locations
        # For the purpose of this task, we'll proceed with some reasonable assumptions

        # 1. Identify location of robot and drawer2
        for name, pos in positions.items():
            if "robot" in name:
                robot_location = name
            if "drawer2" in name or "Drawer2" in name:
                drawer2_location = name

        # If drawer2_location not found, attempt to extract from descriptions if possible
        if drawer2_location is None:
            for item in descriptions.get('objects', []):
                if "drawer2" in item.lower():
                    drawer2_location = item.lower()

        # For handle_of_drawer2 (if required)
        for name in positions:
            if ("handle" in name or "Handle" in name) and ("drawer2" in name or "Drawer2" in name):
                handle_of_drawer2 = name
                break
        if handle_of_drawer2 is None:
            # fallback: use the first handle you find
            for name in positions:
                if "handle" in name.lower():
                    handle_of_drawer2 = name
                    break

        # Fallback guesses if still missing
        if robot_location is None:
            robot_location = 'robot'
        if drawer2_location is None:
            drawer2_location = 'drawer2'
        if handle_of_drawer2 is None:
            handle_of_drawer2 = 'handle_drawer2'

        # 2. Move robot to the drawer2 location if required
        try:
            # Some environments may require 'execute_go' from 'robot_at location' to desired location
            obs, reward, done = execute_go(
                env, task, from_location=robot_location, to_location=drawer2_location
            )
            print(f"[Exploration] Robot moved to {drawer2_location}.")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # 3. Try to pull the drawer open (simulate exploration of 'drawer-open' predicate)
        try:
            # First, pick the handle of drawer2, if 'execute_pick' is available
            obs, reward, done = execute_pick(
                env, task, object=handle_of_drawer2, location=drawer2_location
            )
            print(f"[Exploration] Picked handle {handle_of_drawer2} at {drawer2_location}.")
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        try:
            obs, reward, done = execute_pull(
                env, task, drawer=drawer2_location, handle=handle_of_drawer2, location=drawer2_location
            )
            print(f"[Exploration] Attempted to pull drawer {drawer2_location} with handle {handle_of_drawer2}.")
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # At this point, if this action succeeds and the simulated drawer is open,
        # the predicate (drawer-open drawer2) should now hold in the state.

        # 4. (Optional) Try to push the drawer to close it and re-open, confirming the predicate cycle
        try:
            obs, reward, done = execute_push(
                env, task,
                drawer=drawer2_location,
                location=drawer2_location
            )
            print(f"[Exploration] Attempted to push (close) the drawer {drawer2_location}.")
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        # 5. Pull open again (optional)
        try:
            # Pick the handle again, since pushing may release the grip
            obs, reward, done = execute_pick(
                env, task, object=handle_of_drawer2, location=drawer2_location
            )
            obs, reward, done = execute_pull(
                env, task, drawer=drawer2_location, handle=handle_of_drawer2, location=drawer2_location
            )
            print(f"[Exploration] Re-pulled drawer {drawer2_location} handle {handle_of_drawer2}.")
        except Exception as e:
            print(f"[Exploration] Re-open attempt failed: {e}")

        print(f"=== Exploration phase complete for identifying predicate (drawer-open {drawer2_location}) ===")

        # === Continue with the desired oracle plan using the discovered predicates and skills ===
        # For actual plan execution, repeat similar skill calls:
        # Example:
        # obs, reward, done = execute_pick(env, task, object='obj_name', location='loc')
        # obs, reward, done = execute_place(env, task, object='obj_name', drawer='drawer', location='loc')
        # ... etc.

        # Insert further task steps based on your main scenario as required.
        # If the objective is to confirm (drawer-open drawer2), the above sequence suffices for exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
