# run_skeleton_task.py (Completed Task-Specific Implementation)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills—do not redefine!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ----- Exploration phase for missing predicate/problem -----
        # Feedback tells us: (robot-at drawer_top_place_left)
        # The robot's believed position should be 'drawer_top_place_left', but either:
        #  - the state does not have this, or
        #  - a predicate is missing, or
        #  - exploration is needed to fill in missing predicate info.

        # Here we attempt to move the robot to 'drawer_top_place_left'
        # and verify via the environment state whether we can assert 'robot-at drawer_top_place_left'
        # This is crucial for debugging missing predicates and ensuring PDDL alignment.

        # We need to know the robot's initial location. We'll try to extract it from positions or use a default.
        robot_initial_loc = None
        for key in positions:
            # Heuristic: find location that starts with "robot_" or check initial state info if available
            if "robot" in key and positions[key] is not None:
                robot_initial_loc = key
                break

        # If you have structured observation state, replace this with direct extraction
        # Fallback: assume some default locations if none found
        if robot_initial_loc is None:
            robot_initial_loc = "home_position"

        # Target location for exploration from the feedback:
        target_location = "drawer_top_place_left"

        print(f"[Exploration] Initial robot location: {robot_initial_loc}")
        print(f"[Exploration] Attempting to move robot to: {target_location}")

        try:
            # Call the move/execute_go skill to attempt the transition
            # execute_go(from, to)
            obs, reward, done = execute_go(env, task, from_location=robot_initial_loc, to_location=target_location)
            print(f"[Exploration] Called execute_go from {robot_initial_loc} to {target_location}")
        except Exception as e:
            print(f"[Exploration][ERROR] Could not call execute_go: {e}")

        # Now query or check predicates, or update beliefs about what is missing.
        # The feedback and task suggest the predicate 'robot-at drawer_top_place_left' is not being asserted/used.
        # Exploration is complete here; in an actual robotic system, you might now examine the state, e.g.:
        # check if 'robot-at' predicate or internal state recognizes drawer_top_place_left.

        # ----- (End Exploration Phase) -----

        # TODO: Continue with rest of task's oracle plan here, using available skills
        # For example:
        # - Use execute_pick to pick objects at current location
        # - Use execute_place to place into drawer, etc.
        # - Each step, check state/observation as needed

        # Demo: If we want to do a dummy pick action (only if the corresponding object and location available)
        dummy_obj = None
        for obj in positions:
            if "object" in obj:
                dummy_obj = obj
                break
        if dummy_obj is not None:
            try:
                print(f"[Task] Trying to pick {dummy_obj} at {target_location}")
                obs, reward, done = execute_pick(env, task, object=dummy_obj, location=target_location)
            except Exception as e:
                print(f"[Task][ERROR] Could not execute_pick: {e}")

        # Finish plan as appropriate...

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
