# run_skeleton_task.py (Completed Executable Exploration and Task Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills (do not redefine skill functions)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Skeleton for running exploration and predicate discovery in simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === List available objects, drawers, locations for this env ===
        # For this exploration, we try all combinations
        objects = []
        drawers = []
        handles = []
        locations = []
        for name in positions:
            if name.startswith('obj') or name.startswith('object'):
                objects.append(name)
            if "drawer" in name:
                drawers.append(name)
            if 'handle' in name:
                handles.append(name)
            if "loc" in name or "room" in name or "location" in name or name.startswith("pos"):
                locations.append(name)
        
        # Fallback: if no explicit location names gathered, use all unique parented locations
        if not locations:
            # Try to recover location names from the observation (if available)
            locations = list(set([positions[obj]['location'] for obj in positions if 'location' in positions[obj]]))
            # If still empty, fall back to all keys
            if not locations:
                locations = list(positions.keys())

        # Fallback: for some tasks, robot's current location can be used
        robot_location = None
        if hasattr(task, 'robot_location'):
            robot_location = task.robot_location()
        elif hasattr(task, 'get_robot_location'):
            robot_location = task.get_robot_location()
        else:
            # Search for robot pos in observation dict
            for key in positions:
                if 'robot' in key and 'pos' in positions[key]:
                    robot_location = positions[key]['pos']

        # === Exploration Phase for Missing Predicate Discovery ===

        print("[Exploration] Start: discovering missing predicates via available skills...")

        # List of skill names allowed per instructions
        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Exploration loop: attempt each skill in context to find predicate needed to unlock planning
        # We specifically focus on actions that may reveal a missing predicate, e.g., execute_pull with lock knowledge effect

        exploration_success = False
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        # try picking the object
                        if "execute_pick" in available_skills:
                            print(f"[Exploration] Trying execute_pick with {obj} at {loc}")
                            obs, reward, done = execute_pick(env, task, obj, loc)
                            if done:
                                print(f"[Exploration] Picked {obj} at {loc}.")
                        # try placing the object in the drawer
                        if "execute_place" in available_skills:
                            print(f"[Exploration] Trying execute_place with {obj} in {drawer} at {loc}")
                            obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            if done:
                                print(f"[Exploration] Placed {obj} in {drawer} at {loc}.")
                        # try pulling (unlock knowledge predicate test/discovery)
                        if "execute_pull" in available_skills:
                            for handle in handles:
                                print(f"[Exploration] Trying execute_pull on {drawer} using {handle} at {loc}")
                                obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                                # If an error occurs or this action only works when a predicate (lock-known, drawer-unlocked, etc.) holds, that's key info
                                if done:
                                    print(f"[Exploration] Pulled {drawer} with {handle} at {loc}.")
                                    exploration_success = True
                        if exploration_success:
                            break
                    except Exception as e:
                        print(f"[Exploration] Exception during skill: {e}")
                if exploration_success:
                    break
            if exploration_success:
                break

        if not exploration_success:
            print("[Exploration] Could not determine the missing predicate yet.")
        else:
            print("[Exploration] Possibly missing predicate discovered via pull/open. Review PDDL domain for predicate like lock-known or drawer-unlocked.")

        # === (Optional) Further Steps: Use this info to update planning or report the missing predicate ===
        # Here, we would inform the planning pipeline or domain designer which predicate appears missing (e.g., 'lock-known', 'drawer-unlocked', etc.)
        # End of Exploration

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
