# run_skeleton_task.py (Filled to include exploration for missing predicate discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import * # (All environment skills, do not redefine or reimplement!)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ===
        # EXPLORATION PHASE: Try to identify which predicate is missing, using available skill primitives
        # The idea is: since planner times out, likely an important predicate is missing, such as e.g. "drawer-closed"
        # We'll try acting in the environment using available skills to trigger failures or gather info.

        # This is a generic exploration sequence using only listed available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll try to interact with objects and drawers in various ways and log any error/exception or specific info.
        # For the sake of the example, let's try to:
        # - Move to various locations
        # - Try picking things up
        # - Try opening drawers
        # - Check if 'drawer-closed' state changes as expected

        # The structure assumes that the positions dictionary contains keys for known objects, drawers, handles, and locations.
        # For a fully generic script, we attempt to systematically act using the skills, and report any skill failure (i.e., preconditions not met).

        print("[Exploration] Starting action exploration to probe missing predicate/state requirements.")
        skill_names = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Attempt to go to each location if execute_go is available
        locations = [k for k,v in positions.items() if 'location' in k or 'loc' in k or 'room' in k]
        if len(locations) < 2:
            print("[Exploration] Not enough distinct locations found for movement.")
        else:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                print(f"[Exploration] Attempting to move robot from '{from_loc}' to '{to_loc}'.")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                print("[Exploration] Moved to new location; check preconditions and effects.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # Try to pick up each object-on-floor using execute_pick if available
        object_keys = [k for k in positions.keys() if 'object' in k or 'obj' in k or 'ball' in k]
        for obj_key in object_keys:
            # Try to pick this object at each location
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to pick up {obj_key} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj_key, loc)
                    print("[Exploration] execute_pick success or returned without exception.")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj_key} at {loc}: {e}")

        # Try to interact with drawers (open with execute_pull, close with execute_push) if any handles/drawers defined
        drawer_keys = [k for k in positions if 'drawer' in k]
        handle_keys = [k for k in positions if 'handle' in k]
        for d in drawer_keys:
            # Assume there is a handle associated, and location for testing
            for h in handle_keys:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to hold handle {h}")
                        obs, reward, done = execute_pick(env, task, h, loc)
                        print(f"[Exploration] Holding handle {h}")
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed for handle {h}: {e}")
                    try:
                        print(f"[Exploration] Trying to pull open drawer {d} with handle {h} at {loc}")
                        obs, reward, done = execute_pull(env, task, d, h, loc)
                        print("[Exploration] execute_pull success or returned without exception.")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for drawer {d} handle {h} at {loc}: {e}")
                    try:
                        print(f"[Exploration] Trying to push drawer {d} closed at {loc}")
                        obs, reward, done = execute_push(env, task, d, loc)
                        print("[Exploration] execute_push success or returned without exception.")
                    except Exception as e:
                        print(f"[Exploration] execute_push failed for drawer {d} at {loc}: {e}")

        # Attempt place in all drawers for each object
        for obj_key in object_keys:
            for d in drawer_keys:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying to place {obj_key} in drawer {d} at {loc}")
                        obs, reward, done = execute_place(env, task, obj_key, d, loc)
                        print("[Exploration] execute_place success or returned without exception.")
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj_key} in {d} at {loc}: {e}")

        # Try to sweep each object on floor
        for obj_key in object_keys:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to sweep {obj_key} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj_key, loc)
                    print("[Exploration] execute_sweep success or returned without exception.")
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj_key} at {loc}: {e}")

        # Optionally call rotate/gripper skills; assuming some have parameters, some not
        try:
            print(f"[Exploration] Trying to call execute_rotate (parameters unknown, skipping actual call).")
            # obs, reward, done = execute_rotate(...)
        except Exception as e:
            print(f"[Exploration] execute_rotate call failed: {e}")

        try:
            print(f"[Exploration] Trying to call execute_gripper (no parameters expected).")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] execute_gripper success or returned without exception.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")
        
        print("[Exploration] Finished exploring possible actions. Check log for errors, especially those mentioning state/predicate assumptions.")

        # If specific state or action consistently fails, check if its precondition is missing
        # (e.g., failures to open a drawer if drawer-closed/drawer-open predicates do not seem to update properly.)

        # Further control flow for oracle plan execution can be added here as needed, after exploration phase

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()