# run_skeleton_task.py (Fully Populated for Exploration Phase and Predicate Diagnosis)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined primitive skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Provides a mapping: name -> (x, y, z), or other details as needed
        positions = get_object_positions()
        
        # For exploratory diagnosis — as per feedback: (robot-at drawer-area)
        # We need to try to use skills to reach or confirm the predicate regarding 'robot-at drawer-area'
        # Since the problem involves a missing predicate or its detection about "drawer-area",
        # we will log state before and after attempting to move to or identify the drawer-area

        # --- Candidate locations: try to find 'drawer-area' or the closest relevant key ---
        # Try to use available positions
        # Fallback: scan available keys and look for 'drawer', else print all keys
        drawer_area_key = None
        for key in positions.keys():
            if 'drawer-area' in key:
                drawer_area_key = key
                break
            if 'drawer' in key and drawer_area_key is None:
                drawer_area_key = key  # fallback
        
        print(f"[Exploration] Location keys in environment: {list(positions.keys())}")
        if not drawer_area_key:
            print("[Exploration] Could not find 'drawer-area' in object positions. Using available keys for exploration.")

        # --- Determine initial robot location ---
        # We attempt to infer: from observation or from positions. For this example we just pick a starting key
        robot_start_key = None
        for k in positions.keys():
            if 'robot' in k:
                robot_start_key = k
                break
        # Fallback just pick any key (i.e., first key)
        if robot_start_key is None and len(positions) > 0:
            robot_start_key = list(positions.keys())[0]

        if not robot_start_key or not drawer_area_key:
            print("[Exploration] Not enough location info for robust test. Aborting exploration phase.")
        else:
            # Perform a go (motion) to the 'drawer-area' using execute_go (if possible)
            try:
                from_location = robot_start_key
                to_location = drawer_area_key
                print(f"[Exploration] Attempting to move robot from {from_location} to {to_location} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions[from_location],
                    to_location=positions[to_location]
                )
                print(f"[Exploration] execute_go completed: reward={reward}, done={done}")

                # After move, check for the predicate (robot-at drawer-area) existence/impact
                # (We might try to check obs for this, log it for diagnosis)
                print(f"[Exploration] Observation/state after move: {obs}")

            except Exception as e:
                print(f"[Exploration] Exception during execute_go: {e}")

            # Try an additional skill, e.g., execute_sweep or execute_gripper, to see if predicate changes
            try:
                print(f"[Exploration] Attempting to use execute_gripper at drawer-area.")
                obs2, reward2, done2 = execute_gripper(
                    env,
                    task,
                    target_location=positions[drawer_area_key]
                )
                print(f"[Exploration] execute_gripper completed: reward={reward2}, done={done2}")
                print(f"[Exploration] Observation/state after execute_gripper: {obs2}")

            except Exception as e:
                print(f"[Exploration] Exception during execute_gripper: {e}")

        # --- END OF EXPLORATION PHASE ---
        # Here, the output/logs should give enough information on what is missing w.r.t the feedback
        # This allows devs to diagnose if the 'robot-at drawer-area' predicate is what’s missing from the domain/state

        # You can expand exploration by trying other actions, or calling skills like execute_pick/execution_place,
        # or examine other predicates, mapping their effects before/after, etc.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
