# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skill functions only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ------------------------------------------------------------------
        # EXPLORE: Determine missing predicate (e.g. (drawer-closed ?d - drawer))
        # ------------------------------------------------------------------
        # The feedback indicates that the predicate (drawer-closed ?d) may be missing from the knowledge base
        # Plan: Perform exploration by using available skills to inspect drawer status

        # 1. Identify all drawers and handles
        drawers = [name for name in positions if 'drawer' in name]
        handles = [name for name in positions if 'handle' in name]
        robot_locations = [name for name in positions if 'robot' in name or 'base' in name or 'home' in name]

        # Pick a target drawer to explore (single drawer for simplicity)
        target_drawer = drawers[0] if drawers else None
        target_handle = None
        # Try to find handle corresponding to drawer
        for h in handles:
            # Using handle-of info would require domain introspection;
            # here we assume handle name contains drawer name
            if target_drawer and target_drawer in h:
                target_handle = h
                break
        if not target_handle and handles:
            target_handle = handles[0]

        # Assume initial robot position is robot_locations[0]
        robot_location = robot_locations[0] if robot_locations else None

        print(f"[Explore] Drawers found: {drawers}")
        print(f"[Explore] Handles found: {handles}")
        print(f"[Explore] Robot initial location: {robot_location}")
        print(f"[Explore] Target drawer: {target_drawer}, Target handle: {target_handle}")

        # Try to move to drawer location if needed
        drawer_location = None
        for k, v in positions.items():
            if k == target_drawer:
                drawer_location = v
                break

        # Exploration phase: Try to determine if drawer is closed, open, locked or unlocked
        missing_predicate_detected = False

        try:
            # --- Step 1: Go to drawer ---
            if robot_location and drawer_location:
                # execute_go: from current robot location to drawer location
                print(f"[Explore] Moving robot from '{robot_location}' to position of '{target_drawer}'")
                obs, reward, done = execute_go(env, task, robot_location, target_drawer)
                robot_location = target_drawer  # Update robot current location after move

            # --- Step 2: Try to pick the handle (to open) ---
            # execute_pick: pick handle at location
            if target_handle:
                print(f"[Explore] Attempting to pick up handle '{target_handle}' at drawer '{target_drawer}' location")
                obs, reward, done = execute_pick(env, task, target_handle, target_drawer)

            # --- Step 3: Try to pull to open the drawer ---
            # execute_pull: pull on the drawer using handle, at location
            print(f"[Explore] Attempting to pull drawer '{target_drawer}' using handle '{target_handle}' at location '{target_drawer}'")
            obs, reward, done = execute_pull(env, task, target_drawer, target_handle, target_drawer)

            # If drawer does not open, an exception may be thrown; if so, we suspect missing predicate (locked, closed, etc.)
            print("[Explore] Drawer pulled (opened) successfully: likely not locked and was closed.")
        except Exception as e:
            # Explore by checking if the failure is due to missing precondition or predicate in the model
            print(f"[Explore] Exception encountered during pull/open: {e}")
            print("[Explore] Likely missing predicate relating to drawer's closed or lock status.")

            # Set flag for missing predicate
            missing_predicate_detected = True

        # --- Summarize exploration results ---
        if missing_predicate_detected:
            print("[RESULT] Exploration phase suggests that the predicate (drawer-closed ?d - drawer) may be missing or improperly handled.")
        else:
            print("[RESULT] Exploration phase did NOT indicate missing predicate in current simulation flow.")

        # ------------------------------------------------------------------
        # Continue with normal task plan (manipulation, insertion, etc)
        # ------------------------------------------------------------------
        # For demonstration, let's do a sample plan: pick an object and place it in the now-open drawer

        # Find an object on the floor to pick (not a handle or drawer)
        objects_on_floor = [name for name in positions if 'object' in name and name not in handles + drawers]
        if objects_on_floor:
            object_to_pick = objects_on_floor[0]
            print(f"[Task] Found object '{object_to_pick}' to pick and place in '{target_drawer}'.")

            # Pick up the object
            try:
                obs, reward, done = execute_pick(env, task, object_to_pick, robot_location)
                print(f"[Task] Picked up object '{object_to_pick}'.")

                # Place it into the drawer (should be open now)
                obs, reward, done = execute_place(env, task, object_to_pick, target_drawer, robot_location)
                print(f"[Task] Placed object '{object_to_pick}' into drawer '{target_drawer}'.")

            except Exception as e:
                print(f"[Task] Exception while picking/placing object: {e}")
        else:
            print("[Task] No suitable 'object_on_floor' found for pick and place demonstration.")

        # Optionally, close the drawer at the end
        try:
            obs, reward, done = execute_push(env, task, target_drawer, robot_location)
            print(f"[Task] Drawer '{target_drawer}' closed.")
        except Exception as e:
            print(f"[Task] Exception while closing drawer: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
