# run_skeleton_task.py (Completed for Missing Predicate Exploration and Skill Execution)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions: {'object_name1': (x,y,z), ... 'drawer_X': (x,y,z), ... 'robot': (x,y,z)}

        # === EXPLORATION PHASE ===
        # We need to explore which predicate is missing in the domain/problem setup.
        # According to the exploration knowledge, such missing predicates may relate to whether the lock state (lock-known), object identifications, or temperature is known.
        # We'll use only the provided skills.
        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # None explicitly correspond to the "exploration" predicates (like 'identified', 'lock-known', 'weight-known'), but 'execute_pull' in this domain is a PULLING drawer handle action, not directly a knowledge-action.
        # Thus, we must use 'execute_pull', 'execute_go', and so on, to probe the environment, by triggering preconditions and catching errors or observing effects.

        # In this phase, we try to execute actions whose preconditions/effects may uncover a missing predicate "lock-known", or discover that some predicate is blocking. 
        # We'll attempt various skills in plausible orders; if one fails, we try to determine why.

        # === Generic Object & Location Discovery ===
        # For demonstration we simply look for standard names in positions for drawer, objects, and handles
        # This is just an example; replace according to your simulation's dictionary contents.
        drawer_name = None
        handle_name = None
        object_name = None
        robot_location = None
        for key in positions:
            key_low = key.lower()
            if 'drawer' in key_low and drawer_name is None:
                drawer_name = key
            if 'handle' in key_low and handle_name is None:
                handle_name = key
            if ('obj' in key_low or 'ball' in key_low or 'item' in key_low) and object_name is None:
                object_name = key
            if 'robot' in key_low:
                robot_location = key

        # Fallback if not found
        if not drawer_name:
            for key in positions:
                if key.startswith('drawer'):
                    drawer_name = key
                    break
        if not robot_location:
            robot_location = "robot"  # Default name in many sim environments

        # Extract position tuples
        drawer_pos = positions.get(drawer_name)
        handle_pos = positions.get(handle_name) if handle_name else None
        object_pos = positions.get(object_name)
        robot_pos = positions.get(robot_location)

        # For location parameters, sometimes the environment expects names not coordinates; adjust as necessary.
        location_list = [k for k in positions if 'loc' in k.lower() or 'room' in k.lower() or 'area' in k.lower()]
        if robot_location in positions:
            current_loc = robot_location
        elif location_list:
            current_loc = location_list[0]
        else:
            current_loc = None

        # === Begin Exploration-Execution Sequence ===
        # Try a sequence of skills in a logical order, with exception handling for missing-preconditions (due to missing predicates)
        try:
            print("\n[Exploration] Attempt: Move to object location using execute_go")
            # If needed, first move robot to object location (if not already there)
            if object_name and current_loc:
                obs, reward, done = execute_go(env, task, current_loc, object_name)
                print("[Exploration] Moved to object region:", object_name)
                current_loc = object_name
        except Exception as e:
            print(f"[Exploration] Failed to move to {object_name}: {str(e)}")
        
        # Try to pick up the object, expecting any failure to surface predicates issues
        try:
            print("\n[Exploration] Attempt: Pick up the object with execute_pick")
            # Execute pick: requires (on-floor object), (hand-empty), (robot-free), (robot-at location)
            obs, reward, done = execute_pick(env, task, object_name, current_loc)
            print(f"[Exploration] Successfully picked {object_name}.")
        except Exception as e:
            print(f"[Exploration] Failed to pick {object_name}: {str(e)}")
            # If this fails, perhaps "on-floor" is not true, or hand not empty, or other predicate issue.
        
        # Now, if there is a drawer: try to open it. It requires (holding handle), (handle-of handle drawer), (drawer-unlocked), (drawer-closed), (robot-at location)
        if drawer_name and handle_name:
            try:
                print("\n[Exploration] Attempt: Pick up the handle for drawer opening")
                obs, reward, done = execute_pick(env, task, handle_name, current_loc)
                print(f"[Exploration] Successfully picked up handle {handle_name}.")
            except Exception as e:
                print(f"[Exploration] Failed to pick handle {handle_name}: {str(e)}")
            try:
                print("\n[Exploration] Attempt: Execute drawer pull (open)")
                # This may fail due to missing predicates like drawer-locked/unlocked
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, current_loc)
                print(f"[Exploration] Drawer {drawer_name} opened successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to open drawer {drawer_name}: {str(e)}")
                # If the error message points to missing 'drawer-unlocked' etc., identify those issues.

        # Try to place the object in the drawer if open; this may fail due to preconditions (drawer must be open, not full, etc.)
        try:
            print("\n[Exploration] Attempt: Place object in drawer")
            obs, reward, done = execute_place(env, task, object_name, drawer_name, current_loc)
            print(f"[Exploration] Placed object in drawer {drawer_name}.")
        except Exception as e:
            print(f"[Exploration] Failed to place object in drawer: {str(e)}")

        # Attempt push (close) drawer
        try:
            print("\n[Exploration] Attempt: Close drawer with execute_push")
            obs, reward, done = execute_push(env, task, drawer_name, current_loc)
            print(f"[Exploration] Drawer {drawer_name} closed.")
        except Exception as e:
            print(f"[Exploration] Failed to close drawer {drawer_name}: {str(e)}")

        # Additional available primitive skill explorations (e.g., sweep, gripper)
        try:
            print("\n[Exploration] Attempt: Sweep object (optional exploration)")
            obs, reward, done = execute_sweep(env, task, object_name, current_loc)
            print(f"[Exploration] Swept object {object_name}.")
        except Exception as e:
            print(f"[Exploration] Failed to sweep object: {str(e)}")

        try:
            print("\n[Exploration] Attempt: Gripper reset (optional exploration)")
            obs, reward, done = execute_gripper(env, task)
            print(f"[Exploration] Gripper reset or activated.")
        except Exception as e:
            print(f"[Exploration] Failed gripper action: {str(e)}")

        print("\n[Exploration] Exploration phase complete. Observe logs for info about possible missing predicates.")
        # At this stage, error prints help point out missing predicates (preconditions, etc.) according to the domain.

        # === PLAN EXECUTION (AFTER EXPLORATION) ===
        # Here, after observing which skills/procedures failed, a user can complete a canonical plan to achieve the goal.
        # For this output, we only perform the exploration phase, as required, to reveal missing predicates through skill invocation.
        # If a specific oracle plan is known, insert skill invocations as demonstrated above, step by step.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
