# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided predefined skills only

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve initial object positions (as a dict)
        positions = get_object_positions()
        print("[Exploration] Fetched object positions.")

        # === Exploration Phase for Missing Predicate ===
        # We received feedback that (robot-free) predicate is missing or problematic.
        # We want to verify this in a controlled fashion by attempting skills and inspecting task info.

        # --- 1. Try Execute Pick (expecting requirement: (hand-empty) and (robot-free)) ---
        # Find an object on the floor and its location, also the robot's location
        floor_objs = []
        for obj_name, pos_info in positions.items():
            obj_state = pos_info.get('state', None)
            location = pos_info.get('location', None)
            if obj_state == "on-floor":
                floor_objs.append((obj_name, location))
        robot_loc = None
        for name, pos_info in positions.items():
            if name == "robot":
                robot_loc = pos_info.get('location', None)
                break

        # If no obvious robot location, try inferring (e.g., first room or position)
        if robot_loc is None:
            for name, pos_info in positions.items():
                if pos_info.get('is_robot_here', False):
                    robot_loc = pos_info.get('location')
                    break

        found_missing_predicate = False

        # --- Attempt a pick to see if 'robot-free' matters ---
        if floor_objs and robot_loc:
            obj_name, obj_loc = floor_objs[0]
            print(f"[Exploration] Attempting execute_pick with {obj_name} at {obj_loc}")
            try:
                obs, reward, done = execute_pick(env, task, obj_name, obj_loc)
                print("[Exploration] execute_pick succeeded. Checking if (robot-free) constraint is enforced.")
            except Exception as e:
                error_str = str(e)
                print(f"[Exploration] execute_pick failed with error: {error_str}")
                # Look for missing predicate clue in error (simulate feedback)
                if "robot-free" in error_str or "hand-empty" in error_str:
                    print("[Exploration] Detected missing predicate: (robot-free) likely required!")
                    found_missing_predicate = True
                else:
                    print("[Exploration] Could not confirm missing predicate from this error.")

        # --- If pick did not reveal, try other skills that require robot-free ---
        if not found_missing_predicate:
            # Try to move (execute_go), which usually requires (robot-free)
            locations = []
            for name, pos_info in positions.items():
                if pos_info.get('type', None) == 'location':
                    locations.append(name)
            if len(locations) >= 2:
                from_loc, to_loc = locations[0], locations[1]
                print(f"[Exploration] Attempting execute_go from {from_loc} to {to_loc}")
                try:
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    print("[Exploration] execute_go succeeded. Checking if (robot-free) constraint is enforced.")
                except Exception as e:
                    error_str = str(e)
                    print(f"[Exploration] execute_go failed with error: {error_str}")
                    if "robot-free" in error_str:
                        print("[Exploration] Detected missing predicate: (robot-free) is needed for execute_go!")
                        found_missing_predicate = True

        # --- Summarize Discovery ---
        if found_missing_predicate:
            print("[Exploration] Summary: (robot-free) predicate was found to be critical via testing!")
        else:
            print("[Exploration] Unable to conclusively identify the missing predicate. Manual inspection may be needed.")
        
        # === [After Exploration] Proceed with Main Task (if desired) ===
        # Here, you would implement the task plan using the available skills, now aware of (robot-free).
        # Example: approach an object, pick, open drawer, place, etc.
        # For this template, we just performed the exploration as requested.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()