# run_skeleton_task.py (Completed Executable Code for Exploration Task)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their positions (tuples)
        positions = get_object_positions()

        # --- EXPLORATION PHASE to identify missing predicates
        print("[Exploration] Starting exploration phase to find missing predicates...")

        # This exploration will attempt all skills (actions) on all available objects and locations where feasible,
        # to see what preconditions are missing, as inspired by the exploration domain.
        #
        # Since we do not know the set of objects, drawers, and locations from :objects,
        # we attempt to deduce them with the given 'positions' dictionary.

        # Step 1: Gather candidate object, drawer, and location lists
        # Assume naming conventions are used (e.g., 'drawer1', 'handle1', 'object1', 'location1', etc.)

        # If positions dict is well-formed, use it to detect types.
        all_names = list(positions.keys())
        object_names = [n for n in all_names if 'object' in n or 'ball' in n or 'item' in n]
        drawer_names = [n for n in all_names if 'drawer' in n]
        handle_names = [n for n in all_names if 'handle' in n]
        location_names = [n for n in all_names if 'location' in n or 'loc' in n or 'room' in n]

        # Sometimes object_names or location_names may be empty (if naming is simple),
        # so as fallback, treat all positions as possible locations.
        if not location_names:
            location_names = list(set(all_names) - set(drawer_names) - set(object_names) - set(handle_names))

        # Print debug info about environment entities
        print("[Exploration] Detected objects:", object_names)
        print("[Exploration] Detected drawers:", drawer_names)
        print("[Exploration] Detected handles:", handle_names)
        print("[Exploration] Detected locations:", location_names)

        # Step 2: Systematically attempt each available skill with plausible parameters.
        # Note: For exploration (as in the PDDL exploration domain), we focus on skill preconditions.
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # We will log any failures (exceptions) or unmet precondition errors per skill.
        missing_predicates_log = []

        # Helper to execute and catch exceptions
        def try_execute(skill_fn, *params):
            try:
                print(f"[Exploration] Trying {skill_fn.__name__}({', '.join(map(str, params))})")
                obs, reward, done = skill_fn(env, task, *params)
                print(f"[Exploration] Success: {skill_fn.__name__} with params {params}")
                return True
            except Exception as e:
                print(f"[Exploration] FAILED: {skill_fn.__name__} with params {params}\nError: {e}")
                missing_predicates_log.append((skill_fn.__name__, params, str(e)))
                return False

        # Executing exploration for each skill, using various plausible combinations

        # 1. Try execute_pick for every object at every location
        for obj in object_names:
            for loc in location_names:
                try_execute(execute_pick, obj, loc)

        # 2. Try execute_place for each object in hand, drawer, and location
        # The nature of 'holding' and drawer-open are unknown, so brute force plausible combos.
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try_execute(execute_place, obj, drawer, loc)

        # 3. Try execute_push for each drawer at every location
        for drawer in drawer_names:
            for loc in location_names:
                try_execute(execute_push, drawer, loc)

        # 4. Try execute_pull for each drawer, handle, and location
        if handle_names:
            for drawer in drawer_names:
                for handle in handle_names:
                    for loc in location_names:
                        try_execute(execute_pull, drawer, handle, loc)
        else:
            # Fallback: Try using object_names as handles if handles not found
            for drawer in drawer_names:
                for obj in object_names:
                    for loc in location_names:
                        try_execute(execute_pull, drawer, obj, loc)

        # 5. Try execute_sweep for all objects and locations
        for obj in object_names:
            for loc in location_names:
                try_execute(execute_sweep, obj, loc)

        # 6. Try execute_rotate using all plausible object/location combos, or empty if required
        for obj in object_names:
            for loc in location_names:
                try_execute(execute_rotate, obj, loc)

        # 7. Try execute_go for all possible pairs of locations
        for frm in location_names:
            for to in location_names:
                if frm != to:
                    try_execute(execute_go, frm, to)

        # 8. Try execute_gripper (takes no parameter)
        try:
            print("[Exploration] Trying execute_gripper()")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] Success: execute_gripper()")
        except Exception as e:
            print(f"[Exploration] FAILED: execute_gripper()\nError: {e}")
            missing_predicates_log.append(("execute_gripper", (), str(e)))

        #--------------------------
        # Exploration summary
        print("\n[Exploration] Exploration complete.")
        if missing_predicates_log:
            print("[Exploration] The following skill executions failed, possibly due to missing predicates/preconditions or unmet environment state:")
            for fname, params, errmsg in missing_predicates_log:
                print(f"\tSkill: {fname} | Params: {params} | Error: {errmsg}")
            print("[Exploration] Please review the failed skills above to identify the missing predicates.")
        else:
            print("[Exploration] All skill attempts succeeded; no obvious missing predicates detected.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
