# run_skeleton_task.py (Completed with Exploration Phase for Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skill functions only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Get positions and object dictionaries
        positions = get_object_positions()
        # For this example, we expect the object_positions to return mappings like:
        # positions = {
        #   'robot': { 'name': ..., 'position': ..., 'location': ... },
        #   'drawer_top': {...},
        #   'drawer_handle': {...},
        #   'target_object': {...},
        #   ... }
        #
        # Please ensure your get_object_positions() matches your env objects
        
        # Example names
        robot_location = None
        if 'robot' in positions and 'location' in positions['robot']:
            robot_location = positions['robot']['location']

        drawer_name = 'drawer_top'
        handle_name = 'drawer_handle'
        target_object = None
        for name, info in positions.items():
            if name.startswith('object') or name.startswith('target'):
                target_object = name
                break

        # You may need to adjust object names depending on your environment

        # === Exploration Phase: Attempt to Identify Drawer State Predicates ===
        # The intent is to discover `drawer-unlocked <drawer>` predicate.
        # We use the available skills and try combinations. If skill fails/is blocked, this suggests a missing predicate.
        # We'll attempt to execute a 'pull' (open drawer) sequence and catch exceptions/feedback.

        print("[Exploration] Attempting to open drawer to check lock status.")

        # Step 1: Move robot to the drawer location, if needed
        # Use 'robot_location' and the drawer's location
        drawer_pos = positions.get(drawer_name, {}).get('position', None)
        drawer_loc  = positions.get(drawer_name, {}).get('location', None)
        if drawer_loc is not None and robot_location != drawer_loc:
            try:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_loc}...")
                obs, reward, done = execute_go(env, task, robot_location, drawer_loc)
                robot_location = drawer_loc
            except Exception as e:
                print(f"[Exploration] Failed to move robot: {e}")

        # Step 2: Pick the drawer handle (if available)
        handle_loc = positions.get(handle_name, {}).get('location', None)
        if handle_name in positions and handle_loc == robot_location:
            try:
                print(f"[Exploration] Attempting to pick handle '{handle_name}' at location '{handle_loc}' ...")
                obs, reward, done = execute_pick(env, task, handle_name, robot_location)
            except Exception as e:
                print(f"[Exploration] Failed to pick handle: {e}")

        # Step 3: Attempt to pull the drawer open
        if handle_name in positions:
            try:
                print(f"[Exploration] Attempting to open (pull) '{drawer_name}' using '{handle_name}' ...")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_location)
                print("[Exploration] Drawer pull succeeded. The drawer is likely unlocked.")
            except Exception as e:
                print(f"[Exploration] Drawer pull failed. The drawer may be locked or require unlocking predicate. Error: {e}")

        # Here, in a real system, you might analyze feedback to update the initial state with (drawer-unlocked drawer_top)
        # But in this code, the exploration demonstrates the need for (drawer-unlocked drawer_top) in the state for successful (execute_pull)


        # === Main Oracle Plan Execution (Generic example) ===
        # Now assume the predicate (drawer-unlocked drawer_top) is present so proceed with normal task logic.

        print("[Task] Begin executing main plan: pick target object and place into drawer.")

        # Step 1: Move robot to object location
        if target_object is not None:
            object_loc = positions[target_object].get('location', None)
            if object_loc is not None and robot_location != object_loc:
                try:
                    print(f"[Task] Moving robot to object '{target_object}' at location '{object_loc}'...")
                    obs, reward, done = execute_go(env, task, robot_location, object_loc)
                    robot_location = object_loc
                except Exception as e:
                    print(f"[Task] Failed to move to object: {e}")

            # Step 2: Pick object
            try:
                print(f"[Task] Picking object '{target_object}' at location '{object_loc}' ...")
                obs, reward, done = execute_pick(env, task, target_object, object_loc)
            except Exception as e:
                print(f"[Task] Failed to pick '{target_object}': {e}")

            # Step 3: Move to drawer location
            if drawer_loc is not None and robot_location != drawer_loc:
                try:
                    print(f"[Task] Moving robot to drawer at location '{drawer_loc}' ...")
                    obs, reward, done = execute_go(env, task, robot_location, drawer_loc)
                    robot_location = drawer_loc
                except Exception as e:
                    print(f"[Task] Failed to move to drawer: {e}")

            # Step 4: If drawer is closed, ensure it is open
            # We try to open the drawer again (should succeed if (drawer-unlocked drawer_top))
            handle_loc = positions.get(handle_name, {}).get('location', None)
            if handle_name in positions and handle_loc == robot_location:
                try:
                    print(f"[Task] Picking handle '{handle_name}' at location '{handle_loc}' ...")
                    obs, reward, done = execute_pick(env, task, handle_name, robot_location)
                except Exception as e:
                    print(f"[Task] Failed to pick handle in main plan: {e}")

                try:
                    print(f"[Task] Pulling '{drawer_name}' open using '{handle_name}' ...")
                    obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_location)
                    print(f"[Task] Drawer '{drawer_name}' opened successfully.")
                except Exception as e:
                    print(f"[Task] Failed to pull drawer in main plan: {e}")

            # Step 5: Place the object into the drawer
            try:
                print(f"[Task] Placing object '{target_object}' into drawer '{drawer_name}' at '{drawer_loc}' ...")
                obs, reward, done = execute_place(env, task, target_object, drawer_name, drawer_loc)
                print(f"[Task] Object '{target_object}' placed into drawer '{drawer_name}'.")
            except Exception as e:
                print(f"[Task] Failed to place object into drawer: {e}")

        else:
            print("[Task] No target object found in environment positions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
