# run_skeleton_task.py (Filled in for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Only use these skills; do NOT define your own!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Returns eg. {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === EXPLORATORY PHASE: Detect missing predicates/knowledge via exploration ===
        #
        # Since planner times out, this phase explores with primitives to collect more info,
        # which the planner may need as "pre-discovered" world knowledge.
        # Here, we use only the available primitive skills: 
        #   ['execute_pick', 'execute_place', 'execute_push', 'execute_pull',
        #    'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # The predicates in the exploration PDDL suggest we may need to gather info such as:
        #   - is a location accessible?
        #   - is an object in a certain state? (e.g. locked/unlocked, open/closed)
        #   - is an object identified/known in some way?
        #   - get the lock status of a drawer (e.g. using 'execute_pull')
        #
        # If the planner is stuck (times out), it's likely due to a missing predicate 
        # such as 'drawer-locked', 'drawer-unlocked', or object identification.
        #
        # We'll use available skills to try and interact with objects (esp handles/drawers), 
        # then print/log the result.

        print("[Exploration] Starting environment exploration to discover missing predicates...")

        # For robustness, catch failures on every skill and continue.
        # We'll cycle locations and objects looking for drawers and their handles.

        # The exact keys/structure for objects & positions depend on setup.
        # Let's heuristically try to find all objects with 'drawer' or 'handle' in their names.
        objects = list(positions.keys())
        drawers = [name for name in objects if 'drawer' in name]
        handles = [name for name in objects if 'handle' in name]
        locations = set()
        for obj, pos in positions.items():
            if isinstance(pos, dict) and 'location' in pos:
                locations.add(pos['location'])
            elif isinstance(pos, (list, tuple)) and len(pos) >= 3:
                # Ignore location extraction here - positions might not have location ID directly
                pass

        if not drawers:
            print("[Exploration] No drawers detected in get_object_positions -- skipping drawer exploration.")
        else:
            for drawer in drawers:
                print(f"[Exploration] Investigating drawer: {drawer}")
                # Find handle for this drawer, if any
                handle = None
                for h in handles:
                    if drawer in h or h.replace('handle', '').replace('_', '') in drawer.replace('drawer', '').replace('_', ''):
                        handle = h
                        break
                if handle is None:
                    print(f"[Exploration] No handle found for drawer {drawer}")
                    continue

                # Attempt to pick the handle
                try:
                    print(f"[Exploration] Attempting execute_pick on handle {handle}")
                    # locate the 'location' for the handle, else use the robot base location
                    handle_pos = positions.get(handle)
                    if handle_pos is None:
                        print(f"[Exploration] Handle position for {handle} not found.")
                        continue
                    # Choose the nearest location if available
                    # For illustrative purposes, just call execute_pick with placeholders
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        target_obj=handle,
                        target_pos=handle_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Picked handle {handle}: obs={obs}, reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_pick on {handle}: {e}")

                # Attempt to pull (open) the drawer with the handle
                try:
                    print(f"[Exploration] Attempting execute_pull on drawer {drawer} with handle {handle}")
                    drawer_pos = positions.get(drawer)
                    if drawer_pos is None:
                        print(f"[Exploration] Drawer position for {drawer} not found.")
                        continue
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        target_drawer=drawer,
                        target_handle=handle,
                        target_pos=drawer_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Pulled (attempted to open) drawer {drawer}: obs={obs}, reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_pull on {drawer}: {e}")

        # The above attempts may trigger environment behaviors revealing lock state.
        # If the drawer opens, it's presumably unlocked; if not, it may be locked.
        # This info must be reported (to be used in future planning/grounding).

        print("[Exploration] Exploration phase complete.")

        # === PROBLEM SOLVING: Insert main plan execution code here ===
        #
        # Now, if the missing predicate is determined (e.g., lock state), the environment
        # state can be updated before actual plan execution.
        # The main task/plan logic goes here, e.g.:
        #
        #   for step/action in oracle_plan:
        #       # parse action and arguments
        #       # call the relevant skill function
        #       # handle return/observation
        #
        # Since the plan is not specified, this code simply explores and logs information.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
