# run_skeleton_task.py (Completed with exploration for missing predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # ===== Exploration Phase: Detecting Missing Predicate -----
        # Feedback: (robot-at drawer-area)
        #
        # Our goal here is to explore and verify the existence and behavior of the "robot-at" predicate for the location "drawer-area".
        # Using only available skills, we must perform navigation (execute_go) to "drawer-area" and check the predicates/observations.

        # Plan:
        # 1) Determine current robot location and the "drawer-area" position
        # 2) Use execute_go (or its relevant skill mapping) to move to "drawer-area"
        # 3) Get updated observation, see whether (robot-at drawer-area) holds
        # 4) If robot-at predicate is missing for drawer-area, report/explore

        # -- Step 1: Determine current position and target
        drawer_area_key = None
        for key in positions.keys():
            if 'drawer' in key and 'area' in key:
                drawer_area_key = key
                break
        if drawer_area_key is None:
            # Fallback - try typical location name
            if 'drawer-area' in positions:
                drawer_area_key = 'drawer-area'
            else:
                print("[Exploration] Could not identify drawer-area in object positions!")
                # Optionally, list choices and pick the most likely
                print("Available positions:", positions.keys())
                return

        drawer_area_pos = positions[drawer_area_key]

        # For current position, let's pick one with "robot" or fallback
        robot_start_pos = None
        robot_location_key = None
        for key in positions.keys():
            if 'robot' in key:
                robot_location_key = key
                robot_start_pos = positions[key]
                break
        if robot_start_pos is None:
            # Fallback: just use the first position for exploration
            robot_location_key = list(positions.keys())[0]
            robot_start_pos = positions[robot_location_key]

        print(f"[Exploration] Robot initial at: {robot_location_key} {robot_start_pos}")
        print(f"[Exploration] Attempting to go to: {drawer_area_key} {drawer_area_pos}")

        # -- Step 2: Use execute_go (as per available skills) to move to drawer-area.
        # Mapping execute_go to the skill function:
        # Skill: execute_go(env, task, from_location, to_location, **kwargs)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location_key,
                to_location=drawer_area_key,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved towards {drawer_area_key}")

            # -- Step 3: Obtain new observation (should include predicates or similar info)
            # Here, we attempt to check whether the predicate (robot-at drawer-area) is now true, if such an API exists
            # Alternatively, check observations for related flag
            if hasattr(task, "state_features"):
                # Try to get state in predicate format
                state = task.state_features()
                if any('robot-at' in feature and drawer_area_key in feature for feature in state):
                    print(f"[Exploration Success] (robot-at {drawer_area_key}) is now true in state features.")
                else:
                    print(f"[Exploration] (robot-at {drawer_area_key}) is NOT in state features. The predicate may be missing.")
            else:
                # If the API doesn't support predicate queries, just print we performed exploration
                print("[Exploration] State features unavailable for direct predicate check.")

        except Exception as e:
            print(f"[Exploration] Exception during execute_go to drawer-area: {str(e)}")

        # ======== Task Plan Execution (Example, Replace with Oracle Plan if known) =========
        #
        # Here, after exploration, you would add the steps of the oracle plan using available skill calls:
        # For instance, to pick, place, pull a drawer, etc. (as per your specific task plan).
        # Example (Pseudo-steps, must be replaced by actual plan for your domain/task):
        #
        # try:
        #     obs, reward, done = execute_pick(env, task, object_id, picking_location, ...)
        # except Exception as e:
        #     print("Error during execute_pick:", e)
        #
        # try:
        #     obs, reward, done = execute_pull(env, task, drawer_id, handle_id, drawer_area_key, ...)
        # except Exception as e:
        #     print("Error during execute_pull:", e)
        #
        # (Continue with rest of actions)
        #
        # -- End Plan Steps --

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
