# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing primitive skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Video step wrapping
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # You may use: positions[obj_name] -> (x,y,z)
        print("[Info] Object positions:", positions)

        # === EXPLORATION PHASE ===
        print("[Exploration] Checking for missing predicate: (robot-free)")

        # We will try to execute skills that require (robot-free) in their preconditions and watch for failures.

        # Start with attempting to use every available skill that could be related

        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # To detect missing predicates, we attempt to call 'robot-free' requiring skills and log errors or feedback

        exploration_done = False

        for skill_name in available_skills:
            print(f"[Exploration] Attempting skill: {skill_name}")
            try:
                # We need to select parameters compatible with the function

                if skill_name == 'execute_go':
                    # Movement: try arbitrary locations if available
                    location_names = [n for n in positions if 'location' in n or 'room' in n or 'drawer' in n]
                    # Fallback: get any two different keys
                    loc_list = list(positions.keys())
                    if len(loc_list) >= 2:
                        from_loc = loc_list[0]
                        to_loc = loc_list[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"[Exploration] {skill_name} ran from {from_loc} to {to_loc}.")
                    else:
                        print(f"[Exploration] Not enough locations for {skill_name}.")
                elif skill_name == 'execute_pick':
                    # Look for an object that is on the floor if possible
                    obj_names = [name for name in positions if 'obj' in name or 'object' in name or 'ball' in name or 'handle' in name]
                    loc_names = [name for name in positions if 'location' in name or 'room' in name]
                    if obj_names and loc_names:
                        obj = obj_names[0]
                        loc = loc_names[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"[Exploration] {skill_name} ran for object {obj} at {loc}.")
                elif skill_name == 'execute_place':
                    # Needs holding object, open drawer, etc. Skip since requires complex setup unless obviously possible
                    print(f"[Exploration] Skipping {skill_name}, requires complex state.")
                elif skill_name == 'execute_pull':
                    # Try to find handle and drawer
                    obj_names = [name for name in positions if 'handle' in name]
                    drw_names = [name for name in positions if 'drawer' in name]
                    loc_names = [name for name in positions if 'location' in name or 'room' in name]
                    if obj_names and drw_names and loc_names:
                        # Assume we can manipulate handle, provide dummy input
                        drawer = drw_names[0]
                        handle = obj_names[0]
                        loc = loc_names[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] {skill_name} ran on {drawer}, {handle}, at {loc}.")
                elif skill_name == 'execute_push':
                    # Try to find a drawer and location
                    drw_names = [name for name in positions if 'drawer' in name]
                    loc_names = [name for name in positions if 'location' in name or 'room' in name]
                    if drw_names and loc_names:
                        drawer = drw_names[0]
                        loc = loc_names[0]
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"[Exploration] {skill_name} tried pushing drawer {drawer} at {loc}.")
                elif skill_name == 'execute_sweep':
                    # Try to pick up an object from the floor
                    obj_names = [name for name in positions if 'obj' in name or 'object' in name]
                    loc_names = [name for name in positions if 'location' in name or 'room' in name]
                    if obj_names and loc_names:
                        obj = obj_names[0]
                        loc = loc_names[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"[Exploration] {skill_name} attempted on {obj} at {loc}.")
                elif skill_name == 'execute_gripper':
                    # Call the primitive gripper skill
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] {skill_name} ran.")
                elif skill_name == 'execute_rotate':
                    print(f"[Exploration] Skipping {skill_name}, not defined in this domain.")
                else:
                    print(f"[Exploration] Skill {skill_name} unrecognized.")
            except Exception as e:
                print(f"[Exploration] Exception occurred while running {skill_name}: {repr(e)}")
                # If any error indicates missing precondition 'robot-free', log it
                if "robot-free" in str(e):
                    print("[Exploration] Detected missing predicate: (robot-free)")
                    exploration_done = True
                    break
                else:
                    # Other errors: continue exploration
                    continue

        print("[Exploration] Exploration phase completed.")

        # === TASK EXECUTION PHASE ===
        # After exploration, execute the oracle plan (not provided explicitly).
        # Here you would step-by-step execute the plan using the skill functions
        # and repeat the above style, except with plan-based arguments.

        # Example of skeleton code for oracle plan execution (to be filled with actual plan steps):
        # Steps below are placeholders and should be replaced with actual plan logic.

        # Example:
        # 1. Move to location/room where an object is to be picked
        # 2. Pick the object
        # 3. Move to drawer
        # 4. Open drawer (if unlocked)
        # 5. Place object
        # 6. Close drawer

        # Sample execution (replace parameters with actual ones for your task environment):

        # (Assume object, location, drawer, handle names are present in positions)
        # Example:
        # try:
        #     obs, reward, done = execute_go(env, task, "room_a", "room_b")
        #     obs, reward, done = execute_pick(env, task, "my_object", "room_b")
        #     obs, reward, done = execute_pull(env, task, "my_drawer", "my_handle", "room_b")
        #     obs, reward, done = execute_place(env, task, "my_object", "my_drawer", "room_b")
        #     obs, reward, done = execute_push(env, task, "my_drawer", "room_b")
        # except Exception as e:
        #     print(f"[Task] Exception in oracle plan execution: {repr(e)}")

        print("[Task] Task execution phase (oracle plan) would proceed here.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
