# run_skeleton_task.py (Completed Exploration Phase and Predicate Finder)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only given primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Prepare for Exploration ===
        positions = get_object_positions()

        # === Exploration Phase: Predicate Discovery ===
        # Feedback said: (robot-free) is missing, may need to explore robot-hand/arm status.
        print("[Exploration] Starting predicate exploration.")

        # Simulate simple probe actions to elicit missing predicates:
        # Try picking or placing an object to see what preconditions or failure feedback surfaces.
        exploration_results = {}

        # Try each skill to probe for state outcomes/predicate changes
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]
        probe_obj = None
        probe_drawer = None
        probe_loc = None
        probe_handle = None

        # Attempt to locate objects, locations, and drawer/handles for probe actions
        # Assume the positions dict includes: objects, drawers, locations, handles
        for k in positions:
            if 'drawer' in k and probe_drawer is None:
                probe_drawer = k
            elif 'handle' in k and probe_handle is None:
                probe_handle = k
            elif 'object' in k and probe_obj is None:
                probe_obj = k
            elif probe_loc is None:
                probe_loc = k

        # Fallback: assign any available key for probe_obj/probe_loc if can't autodetect
        if probe_obj is None and len(positions) > 0:
            probe_obj = list(positions.keys())[0]
        if probe_loc is None and len(positions) > 0:
            probe_loc = list(positions.keys())[0]

        # 1. Try to pick up an object directly
        try:
            print(f"[Exploration] Attempting execute_pick({probe_obj}, {probe_loc})")
            obs_pick, reward_pick, done_pick = execute_pick(
                env, task, obj=probe_obj, loc=probe_loc
            )
            exploration_results['execute_pick'] = (obs_pick, reward_pick, done_pick)
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        # 2. Try to place the object (requires complex state; may fail if not holding)
        if probe_drawer is not None:
            try:
                print(f"[Exploration] Attempting execute_place({probe_obj}, {probe_drawer}, {probe_loc})")
                obs_place, reward_place, done_place = execute_place(
                    env, task, obj=probe_obj, drawer=probe_drawer, loc=probe_loc
                )
                exploration_results['execute_place'] = (obs_place, reward_place, done_place)
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # 3. Try to pull (open) drawer (needs handle-of, holding handle, etc.)
        if probe_drawer is not None and probe_handle is not None:
            try:
                print(f"[Exploration] Attempting execute_pull({probe_drawer}, {probe_handle}, {probe_loc})")
                obs_pull, reward_pull, done_pull = execute_pull(
                    env, task, drawer=probe_drawer, handle=probe_handle, loc=probe_loc
                )
                exploration_results['execute_pull'] = (obs_pull, reward_pull, done_pull)
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try to push (close) drawer
        if probe_drawer is not None:
            try:
                print(f"[Exploration] Attempting execute_push({probe_drawer}, {probe_loc})")
                obs_push, reward_push, done_push = execute_push(
                    env, task, drawer=probe_drawer, loc=probe_loc
                )
                exploration_results['execute_push'] = (obs_push, reward_push, done_push)
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # 5. Try to go (move) from one location to another (if possible)
        keys = list(positions.keys())
        if len(keys) > 1:
            from_loc = keys[0]
            to_loc = keys[1]
            try:
                print(f"[Exploration] Attempting execute_go({from_loc}, {to_loc})")
                obs_go, reward_go, done_go = execute_go(env, task, from_loc=from_loc, to_loc=to_loc)
                exploration_results['execute_go'] = (obs_go, reward_go, done_go)
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # Additional exploration actions as needed
        try:
            print(f"[Exploration] Attempting execute_sweep({probe_obj}, {probe_loc})")
            obs_sweep, reward_sweep, done_sweep = execute_sweep(env, task, obj=probe_obj, loc=probe_loc)
            exploration_results['execute_sweep'] = (obs_sweep, reward_sweep, done_sweep)
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        try:
            print(f"[Exploration] Attempting execute_gripper()")
            obs_gripper, reward_gripper, done_gripper = execute_gripper(env, task)
            exploration_results['execute_gripper'] = (obs_gripper, reward_gripper, done_gripper)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Completed. Results (may contain None if failed):")
        print(exploration_results)

        # === Determine Missing Predicate ===
        # Analyzing feedback and exploration, the missing predicate is likely (robot-free).
        # It's required as a precondition in pick, go, sweep, gripper, push actions.
        # Take note of failures like "robot is not free" or "hand is full" to pinpoint missing information.

        print("[Exploration] The likely missing predicate for planning is: (robot-free)")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
