# run_skeleton_task.py (Completed for Exploration Phase to Find Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Skill primitives only, as per instructions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing simulation
        init_video_writers(obs)

        # Wrap task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example structure: {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase to Find the Missing Predicate ===
        # As per feedback and knowledge: find which predicate is missing (e.g., identified, temperature-known, weight-known, durability-known, lock-known)
        # To do this, try invoking every available exploration skill and observe the effects or errors.

        # Step 1: Attempt every available primitive skill from the provided skills,
        # focused on exploring which action (and thus which predicate) causes the planning system to get "unstuck".

        # These are the available skills per input:
        exploration_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Get explicit lists of objects, drawers, handles, locations from the current environment state.
        # Since we don't have the full (:objects ...) nor inferior details, we fetch what we can.
        # We'll attempt to operate on all keys retrievable from get_object_positions().
        # If the environment or task exposes methods or attributes for object types, we would use those instead.

        object_names = [name for name in positions if "object" in name or "obj" in name]
        drawer_names = [name for name in positions if "drawer" in name]
        handle_names = [name for name in positions if "handle" in name]
        location_names = [name for name in positions if "room" in name or "loc" in name or "location" in name]

        # Fallback if none found:
        if not location_names:
            location_names = ["loc1", "loc2"]

        # 1. Try execute_pick on every object-floor-location combination
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting execute_pick(obj={obj}, loc={loc})")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    print(f"[Exploration] Success execute_pick({obj}, {loc}) -> reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # 2. Try execute_place for each object/drawer/location combination
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting execute_place(obj={obj}, drawer={drawer}, loc={loc})")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] Success execute_place({obj}, {drawer}, {loc}) -> reward={reward}, done={done}")
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj}, {drawer} at {loc}: {e}")

        # 3. Try execute_push and execute_pull on each drawer/location/handle
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting execute_push(drawer={drawer}, loc={loc})")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    print(f"[Exploration] Success execute_push({drawer}, {loc}) -> reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")

        for drawer in drawer_names:
            for handle in handle_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting execute_pull(drawer={drawer}, handle={handle}, loc={loc})")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] Success execute_pull({drawer}, {handle}, {loc}) -> reward={reward}, done={done}")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer}, {handle} at {loc}: {e}")

        # 4. Try execute_sweep and execute_rotate on every object/location
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting execute_sweep(obj={obj}, loc={loc})")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    print(f"[Exploration] Success execute_sweep({obj}, {loc}) -> reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

                try:
                    print(f"[Exploration] Attempting execute_rotate(obj={obj}, loc={loc})")
                    obs, reward, done = execute_rotate(env, task, obj, loc)
                    print(f"[Exploration] Success execute_rotate({obj}, {loc}) -> reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed for {obj} at {loc}: {e}")

        # 5. Try execute_go between all pairs of distinct locations
        for i in range(len(location_names)):
            for j in range(len(location_names)):
                if i != j:
                    from_loc = location_names[i]
                    to_loc = location_names[j]
                    try:
                        print(f"[Exploration] Attempting execute_go(from={from_loc}, to={to_loc})")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"[Exploration] Success execute_go({from_loc}, {to_loc}) -> reward={reward}, done={done}")
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # 6. Try execute_gripper if available
        try:
            print(f"[Exploration] Attempting execute_gripper()")
            obs, reward, done = execute_gripper(env, task)
            print(f"[Exploration] Success execute_gripper, reward={reward}, done={done}")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("===== EXPLORATION PHASE COMPLETED: Review the printed outputs to determine which action (predicate) was missing or caused a logical block =====")

        # At this point, review which action(s) succeeded or failed unexpectedly to deduce the missing predicate(s).
        # If planning continues to time out, further targeted exploration required based on known failure points.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()