# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All primitive skill functions available

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video recording if needed
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object/robot/drawer positions & other meta info
        positions = get_object_positions()  # {'robot': ..., 'drawer1': ..., ...}

        # === EXPLORATION PHASE (Predicate Discovery) ===
        # To ensure (drawer-unlocked drawer1) is known, we must explore drawer lock state.
        # This phase tries available "exploration" skills to discover the missing predicate.
        # In this case: Try to pull the drawer to check if it's locked/unlocked.
        # We'll use only predefined skills.

        print("[Exploration] Attempting to discover drawer lock predicate...")

        # Example variable names, these should be replaced with actual ones from positions or task interface.
        robot_pos = positions.get("robot", None)
        drawer1_pos = positions.get("drawer1", None)
        handle1_pos = positions.get("handle1", None)
        graspable_handles = [k for k in positions if "handle" in k]

        # Heuristically infer names (common in RLBench simulation)
        drawer_name = None
        handle_name = None
        for k in positions:
            if "drawer" in k:
                drawer_name = k
            if "handle" in k:
                handle_name = k

        # If not found, fallback to plausible names
        if drawer_name is None:
            drawer_name = "drawer1"
        if handle_name is None and graspable_handles:
            handle_name = graspable_handles[0]
        elif handle_name is None:
            handle_name = "handle1"

        # Assume the robot can move next to the drawer/handle
        try:
            # 1. Move robot to handle location
            if robot_pos is not None and positions.get(handle_name) is not None:
                from_loc = robot_pos
                to_loc = positions[handle_name]
                print(f"[Exploration] Moving robot from {from_loc} to {to_loc} (handle location).")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc
                )
            else:
                print("[Exploration] Cannot infer robot or handle position. Skipping move.")

            # 2. Try to pick the handle (robot's hand must be empty and at handle position)
            print("[Exploration] Attempting to pick the handle.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=to_loc
            )
            # 3. Try to pull the drawer (this may discover the lock state, i.e., (drawer-unlocked ...))
            print("[Exploration] Attempting to pull the drawer to determine lock state.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=to_loc
            )
            # After this, the environment should have knowledge about the predicate (drawer-unlocked ...).
            print("[Exploration] Drawer lock predicate discovery complete.")
        except Exception as e:
            print(f"[Exploration] Error in exploration phase: {e}")

        # === MAIN TASK PHASE ===
        # At this point, the system should know if (drawer-unlocked drawer1) holds.
        # Now, continue with the main skills as per the oracle plan (example: open the drawer, pick/place, etc.)

        # The following steps are generic and must be adapted as per the actual task at hand.
        # Example plan:
        #   1) Go to the drawer
        #   2) Pick the handle
        #   3) Pull (open) the drawer
        #   4) Pick an object from the floor
        #   5) Place it in the drawer
        #   6) Close the drawer

        try:
            # 1. Ensure robot is at the drawer location
            if robot_pos is not None and positions.get(drawer_name) is not None:
                from_loc = positions[handle_name]
                to_loc = positions[drawer_name]
                print("[Plan] Moving robot to drawer location...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc
                )
            else:
                print("[Plan] Cannot infer robot or drawer position; skipping this move.")

            # 2. Open the drawer by pulling handle (assuming handle is already picked)
            print("[Plan] Ensuring handle is held and pulling (open) the drawer.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=to_loc
            )

            # 3. Pick up an object from the floor (choose first available)
            floor_objects = [k for k in positions if "object" in k or "ball" in k]
            chosen_object = floor_objects[0] if floor_objects else None
            if chosen_object is not None:
                obj_pos = positions[chosen_object]
                print(f"[Plan] Picking object {chosen_object} from {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=chosen_object,
                    location=obj_pos
                )
            else:
                print("[Plan] No floor object found to pick.")

            # 4. Place object into the drawer
            print(f"[Plan] Attempting to place object {chosen_object} into drawer {drawer_name}.")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=chosen_object,
                drawer_name=drawer_name,
                location=to_loc
            )

            # 5. Close the drawer (push)
            print(f"[Plan] Closing the drawer {drawer_name}.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=to_loc
            )

        except Exception as e:
            print(f"[Plan] Exception during main plan execution: {e}")

        print("[Task] Oracle plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
