# run_skeleton_task.py (Filled with Exploration Phase to Find Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only these predefined primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now including an exploration phase to check for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Positions of objects and scene elements.
        try:
            positions = get_object_positions()  # dictionary: key = object name, value = (x, y, z)
        except Exception as e:
            print("[Warning] get_object_positions() failed:", e)
            positions = {}

        # === PHASE 1: Exploration Phase for Missing Predicate ===
        # The goal: try available exploration skills and see which predicate might be missing
        #
        # Since feedback suggests the problem is a missing predicate needed for planning,
        # we try exploration-related available skills to see which ones provide new state predicates.
        #
        # For this, we try each relevant exploration skill with plausible parameters.
        # We log exceptions so the process is robust.

        print("[Exploration] Starting to probe for missing predicates with available exploration actions.")

        exploration_skills = [
            'execute_sweep',      # on-floor → ?
            'execute_gripper',    # robot-free → ?
            # No 'execute_go_identify', 'execute_go_temperature', or 'execute_pick_weight' in available skills, so skip those.
        ]

        # Try exploration skills with all objects/locations (best effort)
        for skill in exploration_skills:
            for obj_name in positions.keys():
                # Assuming objects names map to on-floor objects
                obj_pos = positions[obj_name]
                location_name = None
                # Try to infer location from object position
                for loc_candidate in positions.keys():
                    if 'location' in loc_candidate.lower():
                        location_name = loc_candidate
                        break
                # If no explicit location, just use the first object as location proxy for demo
                if location_name is None and len(positions) > 0:
                    location_name = list(positions.keys())[0]
                # Call the exploration skill
                try:
                    if skill == 'execute_sweep':
                        print(f"[Exploration] Attempt skill: {skill} on {obj_name} at {location_name}")
                        obs, reward, done = execute_sweep(env, task, obj_name, location_name)
                        if done:
                            print(f"[Exploration] {skill} led to completion (done=True) on {obj_name} at {location_name}")
                            break
                    elif skill == 'execute_gripper':
                        print(f"[Exploration] Attempt skill: {skill}")
                        obs, reward, done = execute_gripper(env, task)
                        if done:
                            print("[Exploration] Exploration (+execute_gripper) signaled done")
                            break
                except Exception as e:
                    print(f"[Exploration] Exception during {skill} on {obj_name}: {e}")

        # === PHASE 2: Task Plan Execution ===
        # After exploration, proceed with main plan using the available skills.

        # Example logic: suppose the plan is to pick an object, open a drawer, and place the object in it.
        # All actions must call only the pre-defined skills.
        # We'll use the available skill names as a reference for possible actions,
        # and infer hypothetical object and location names for demonstration.

        # Plan Step 1: Move to location (execute_go)
        try:
            possible_locations = [name for name in positions.keys() if "loc" in name.lower() or "room" in name.lower() or "area" in name.lower()]
            if len(possible_locations) >= 2:
                from_location = possible_locations[0]
                to_location = possible_locations[1]
            else:
                from_location, to_location = "location1", "location2"
            print(f"[Plan] Moving from {from_location} to {to_location}")
            obs, reward, done = execute_go(env, task, from_location, to_location)
            if done:
                print("[Plan] Task completed after execute_go!")
                return
        except Exception as e:
            print(f"[Plan] Exception during execute_go: {e}")

        # Plan Step 2: Pick up an object on the floor (execute_pick)
        try:
            # Choose the first object as the one to pick, at target location
            object_names = list(positions.keys())
            if len(object_names) > 0:
                pick_object = object_names[0]
            else:
                pick_object = "object1"
            print(f"[Plan] Picking up {pick_object} at {to_location}")
            obs, reward, done = execute_pick(env, task, pick_object, to_location)
            if done:
                print("[Plan] Task completed after execute_pick!")
                return
        except Exception as e:
            print(f"[Plan] Exception during execute_pick: {e}")

        # Plan Step 3: Pull open a drawer (execute_pull)
        try:
            # Find a drawer-handle pair if possible
            drawer_name = None
            handle_name = None
            for name in positions.keys():
                if "drawer" in name.lower():
                    drawer_name = name
                if "handle" in name.lower():
                    handle_name = name
            if drawer_name is None:
                drawer_name = "drawer1"
            if handle_name is None:
                handle_name = pick_object  # fallback
            print(f"[Plan] Pulling open {drawer_name} using {handle_name} at {to_location}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, to_location)
            if done:
                print("[Plan] Task completed after execute_pull!")
                return
        except Exception as e:
            print(f"[Plan] Exception during execute_pull: {e}")

        # Plan Step 4: Place picked object in drawer (execute_place)
        try:
            print(f"[Plan] Placing {pick_object} in {drawer_name} at {to_location}")
            obs, reward, done = execute_place(env, task, pick_object, drawer_name, to_location)
            if done:
                print("[Plan] Task completed after execute_place!")
                return
        except Exception as e:
            print(f"[Plan] Exception during execute_place: {e}")

        # Plan Step 5: Push drawer closed (execute_push)
        try:
            print(f"[Plan] Pushing drawer {drawer_name} closed at {to_location}")
            obs, reward, done = execute_push(env, task, drawer_name, to_location)
            if done:
                print("[Plan] Task completed after execute_push!")
                return
        except Exception as e:
            print(f"[Plan] Exception during execute_push: {e}")

        print("[Task] Plan execution phase completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
