# run_skeleton_task.py (COMPLETED for EXPLORATION and PLAN EXECUTION)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # --- Environment Setup ---
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # --- Retrieve Object Positions ---
        # This returns a dictionary: name -> (position tuple)
        positions = get_object_positions()

        # --- EXPLORATION PHASE: Find Missing Predicate ---
        #
        # Feedback indicates: (robot-at drawer-area)
        # But 'robot-at' is not available in the task domain, only in the exploration domain.
        # Goal: Explore the environment using only the available skills to determine the equivalent knowledge.

        # We'll attempt to move/go to all known object/area keys from the initial robot position,
        # and check if we can reach/recognize 'drawer-area' or similar predicates.

        robot_start_location = None
        # Try to determine the robot's initial location from observation if possible
        # Otherwise, default to first location key found
        for name in positions:
            if 'robot' in name or 'base' in name:
                robot_start_location = name
                break
        if robot_start_location is None:
            # Fallback: use a hardcoded name or the first key
            robot_start_location = list(positions.keys())[0]

        current_location = robot_start_location

        # Find any keys that look like locations or area
        location_keys = []
        for k in positions:
            if 'area' in k or 'zone' in k or 'room' in k or 'drawer' in k or 'table' in k:
                location_keys.append(k)
        location_keys = list(set(location_keys))

        # Record which keys were successful for navigation
        successfully_reached = []
        for loc in location_keys:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Attempting to move from {current_location} to {loc}")
                obs, reward, done = execute_go(env, task, from_location=current_location, to_location=loc)
                if reward is not None:
                    print(f"[Exploration] Moved to {loc}; checking if (robot-at {loc}) holds.")
                else:
                    print(f"[Exploration] No explicit reward at {loc}.")
                successfully_reached.append(loc)
                current_location = loc  # update for next iteration
            except Exception as e:
                print(f"[Exploration] Could not move to {loc}: {e}")

        # The missing predicate suggested by feedback is (robot-at drawer-area)
        # So we need to confirm if the area named 'drawer-area' (or similar) exists.
        # If not, 'drawer' keys may be the intended match.

        # --- PLAN EXECUTION: Example Use of Primitive Skills ---

        # For demonstration, implement a generic sample plan that approaches an object and interacts
        # According to the available skills: 
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to find an object to pick up (preferably on the floor)
        object_to_pick = None
        pick_location = None
        for obj_name, pos in positions.items():
            if 'object' in obj_name or 'item' in obj_name or 'handle' in obj_name:
                object_to_pick = obj_name
                # Try to associate the object with a location (using proximity or naming)
                for loc_key in location_keys:
                    if loc_key in obj_name:
                        pick_location = loc_key
                        break
                if not pick_location and len(location_keys) > 0:
                    pick_location = location_keys[0]
                break

        if object_to_pick and pick_location:
            # Move to pick_location if not already there
            if current_location != pick_location:
                try:
                    print(f"[Task] Moving from {current_location} to {pick_location} for picking {object_to_pick}.")
                    obs, reward, done = execute_go(env, task, from_location=current_location, to_location=pick_location)
                    current_location = pick_location
                except Exception as e:
                    print(f"[Task] Failed to move to {pick_location}: {e}")

            # Pick up the object
            try:
                print(f"[Task] Picking up {object_to_pick} at {pick_location}.")
                obs, reward, done = execute_pick(env, task, object_name=object_to_pick, location=pick_location)
            except Exception as e:
                print(f"[Task] Failed to pick {object_to_pick}: {e}")
        else:
            print("[Task] Could not determine a suitable object and location for pick.")

        # 2. Try opening a drawer if there is one
        drawer_name = None
        handle_name = None
        for key in positions:
            if 'drawer' in key and 'area' not in key:
                drawer_name = key
            if 'handle' in key:
                handle_name = key
        if drawer_name and handle_name:
            # Assume drawer and handle are at current location for simplicity
            try:
                print(f"[Task] Attempting to pull open {drawer_name} via {handle_name} at {current_location}.")
                obs, reward, done = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location=current_location)
            except Exception as e:
                print(f"[Task] Could not pull open {drawer_name}: {e}")
        else:
            print("[Task] Could not detect both a drawer and handle.")

        # 3. Try placing the object in the drawer
        if object_to_pick and drawer_name and pick_location:
            try:
                print(f"[Task] Placing {object_to_pick} in {drawer_name} at {pick_location}.")
                obs, reward, done = execute_place(env, task, object_name=object_to_pick, drawer_name=drawer_name, location=pick_location)
            except Exception as e:
                print(f"[Task] Failed to place {object_to_pick}: {e}")

        # 4. Try closing the drawer
        if drawer_name and pick_location:
            try:
                print(f"[Task] Pushing (closing) {drawer_name} at {pick_location}.")
                obs, reward, done = execute_push(env, task, drawer_name=drawer_name, location=pick_location)
            except Exception as e:
                print(f"[Task] Could not push (close) {drawer_name}: {e}")

        # (If you want to expand, can iterate other available skills as needed...)

        print("[Exploration] These locations were successfully reached during exploration:")
        for loc in successfully_reached:
            print(f"- {loc}")

        if 'drawer-area' in successfully_reached:
            print("[Exploration] Confirmed that (robot-at drawer-area) is a valid predicate after exploration.")

        # Exploration phase ended - proceed with additional plan steps if needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
