# run_skeleton_task.py (Completed with Exploration Logic & Only Existing Skills)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine skill functions, use as provided

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Run task with exploration for missing predicates using existing skills only.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Exploration] Initial object positions:", positions)

        # === Exploration Phase (for missing predicate) ===
        # The system failed to plan due to a missing predicate.
        # We now perform skill-based exploration step-by-step to try to elicit information relating to lock or other unknown object properties.
        # Only use available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assumptions:
        # - Object and drawer names can be inferred from the positions dict and observation
        # - Try using 'execute_pull' to probe for lock status of drawers/handles per the exploration domain suggestion
        # - 'execute_go' for moving
        # - Use try/except to gracefully capture failures that may suggest missing predicates

        # Identify all objects, drawers, locations from positions/obs
        object_names = []
        drawer_names = []
        location_names = []
        for name in positions.keys():
            if "drawer" in name:
                drawer_names.append(name)
            elif "loc" in name or "table" in name or "zone" in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # If we cannot detect types by names, fallback to defaults if possible
        if not location_names:
            location_names = ["table", "area_1", "area_2"]
        if not drawer_names:
            drawer_names = ["drawer_1", "drawer_2"]
        if not object_names:
            object_names = ["object_1", "object_2"]

        # Choose handles: assume objects with 'handle' in name
        handle_names = [name for name in object_names if "handle" in name]
        if not handle_names:
            handle_names = ["handle_1"]

        # Exploration steps
        exploration_done = False
        for drawer in drawer_names:
            # Try to find a handle for this drawer
            candidate_handles = [h for h in handle_names if h in positions or True]  # Weak check: use handle_1
            handle = candidate_handles[0] if candidate_handles else handle_names[0]
            for loc in location_names:
                print(f"[Exploration] Trying to execute_pick(handle={handle}, loc={loc}) for drawer={drawer}")
                try:
                    # Attempt to pick handle
                    obs, reward, done = execute_pick(env, task, handle, loc)
                    print(f"  [Done] execute_pick: handle={handle} loc={loc} --> reward={reward}, done={done}")
                    if done:
                        print(f"  [Early Termination] Done after pick {handle} at {loc}")
                        exploration_done = True
                        break
                except Exception as e:
                    print(f"  [Exception] execute_pick failed for handle={handle} at {loc}: {e}")

                print(f"[Exploration] Trying to execute_pull(drawer={drawer}, handle={handle}, loc={loc})")
                try:
                    # Attempt to pull the drawer with the handle (simulate lock-known info)
                    obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                    print(f"  [Done] execute_pull: drawer={drawer} handle={handle} loc={loc} --> reward={reward}, done={done}")
                    if done:
                        print(f"  [Early Termination] Done after pull {drawer} at {loc}")
                        exploration_done = True
                        break
                except Exception as e:
                    print(f"  [Exception] execute_pull failed for drawer={drawer} handle={handle} at {loc}: {e}")

                print(f"[Exploration] Trying to execute_push(drawer={drawer}, loc={loc})")
                try:
                    # Attempt to push (close) the drawer
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    print(f"  [Done] execute_push: drawer={drawer} loc={loc} --> reward={reward}, done={done}")
                    if done:
                        print(f"  [Early Termination] Done after push {drawer} at {loc}")
                        exploration_done = True
                        break
                except Exception as e:
                    print(f"  [Exception] execute_push failed for drawer={drawer} at {loc}: {e}")

            if exploration_done:
                break

        # Try object pick & place to see if any object-specific predicate is missing
        if not exploration_done:
            for obj in object_names:
                for loc in location_names:
                    print(f"[Exploration] Trying to execute_pick(obj={obj}, loc={loc})")
                    try:
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"  [Done] execute_pick: obj={obj} loc={loc} --> reward={reward}, done={done}")
                        if done:
                            print(f"  [Early Termination] Done after pick {obj} at {loc}")
                            exploration_done = True
                            break
                    except Exception as e:
                        print(f"  [Exception] execute_pick failed for obj={obj} at {loc}: {e}")

                    for drawer in drawer_names:
                        print(f"[Exploration] Trying to execute_place(obj={obj}, drawer={drawer}, loc={loc})")
                        try:
                            obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            print(f"  [Done] execute_place: obj={obj} drawer={drawer} loc={loc} --> reward={reward}, done={done}")
                            if done:
                                print(f"  [Early Termination] Done after place {obj} in {drawer} at {loc}")
                                exploration_done = True
                                break
                        except Exception as e:
                            print(f"  [Exception] execute_place failed for obj={obj} drawer={drawer} at {loc}: {e}")
                    if exploration_done:
                        break
                if exploration_done:
                    break

        # Optionally try gripper, sweep or rotate if those skills may reveal more info about predicates
        if not exploration_done:
            print(f"[Exploration] Trying execute_gripper()")
            try:
                obs, reward, done = execute_gripper(env, task)
                print(f"  [Done] execute_gripper --> reward={reward}, done={done}")
            except Exception as e:
                print(f"  [Exception] execute_gripper failed: {e}")

            print(f"[Exploration] Trying execute_sweep on {object_names[0]} at {location_names[0]}")
            try:
                obs, reward, done = execute_sweep(env, task, object_names[0], location_names[0])
                print(f"  [Done] execute_sweep: obj={object_names[0]} loc={location_names[0]} --> reward={reward}, done={done}")
            except Exception as e:
                print(f"  [Exception] execute_sweep failed: {e}")

            if "execute_rotate" in globals():
                print(f"[Exploration] Trying execute_rotate (if implemented)")
                try:
                    obs, reward, done = execute_rotate(env, task)
                    print(f"  [Done] execute_rotate --> reward={reward}, done={done}")
                except Exception as e:
                    print(f"  [Exception] execute_rotate failed: {e}")

        print("[Exploration] Exploration finished. Review the console logs above to identify which predicate may be missing from PDDL and blocking the plan.")

        # === End task here (no oracle plan available until missing predicate is fixed) ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
