# run_skeleton_task.py (Completed according to instructions)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All skill functions available

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Complete skeleton that executes exploration phase to find missing predicates using only available skills.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # === EXPLORATION PHASE: Attempt to find the missing predicate ===
        # Based on your exploration domain, we should attempt to call all available SKILLS
        # in the env to check for precondition failures that may relate to missing predicates.

        # We will:
        #   1) Loop over the available skills and attempt to execute them in a likely-safe fashion
        #   2) Catch any precondition errors and print out what failed for diagnostics
        #   3) Log any state changes for debugging
        # No skills are invented here -- we only call pre-existing skill functions.

        # You may need to adapt the following for your own object/location names:
        all_objects = [k for k,v in positions.items() if not k.startswith('robot') and not 'drawer' in k and not 'handle' in k]
        all_locations = [k for k,v in positions.items() if 'location' in k or k.startswith('loc')]
        all_drawers = [k for k,v in positions.items() if 'drawer' in k]
        all_handles = [k for k,v in positions.items() if 'handle' in k]

        # Try to choose some plausible items
        obj = all_objects[0] if all_objects else None
        drawer = all_drawers[0] if all_drawers else None
        handle = all_handles[0] if all_handles else None
        loc = all_locations[0] if all_locations else None
        loc2 = all_locations[1] if len(all_locations)>1 else None

        # Ensure robot location is known for movement
        robot_pos = None
        for k in positions:
            if "robot" in k and "location" in k:
                robot_pos = k
                if not loc:
                    loc = k

        print("[Exploration] Identified objects:", all_objects)
        print("[Exploration] Identified drawers:", all_drawers)
        print("[Exploration] Identified handles:", all_handles)
        print("[Exploration] Identified locations:", all_locations)
        print("[Exploration] Robot location:", robot_pos)

        # Available skill list:
        skill_names = [
            'execute_pick', 
            'execute_place', 
            'execute_push', 
            'execute_pull', 
            'execute_sweep', 
            'execute_rotate', 
            'execute_go', 
            'execute_gripper'
        ]
        
        # Try each skill in succession and print results; gather info for missing predicates
        for skill in skill_names:
            try:
                print(f"\n[Exploration] Trying skill: {skill}")
                # Each function may require specific arguments based on domain PDDL
                if skill == "execute_pick":
                    if obj and loc:
                        print(f"Calling {skill}({obj}, {loc})")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                    else:
                        print(f"[Exploration] Skipping {skill} -- missing object or location")
                elif skill == "execute_place":
                    if obj and drawer and loc:
                        print(f"Calling {skill}({obj}, {drawer}, {loc})")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    else:
                        print(f"[Exploration] Skipping {skill} -- missing object, drawer, or location")
                elif skill == "execute_push":
                    if drawer and loc:
                        print(f"Calling {skill}({drawer}, {loc})")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                    else:
                        print(f"[Exploration] Skipping {skill} -- missing drawer or location")
                elif skill == "execute_pull":
                    if drawer and handle and loc:
                        print(f"Calling {skill}({drawer}, {handle}, {loc})")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                    else:
                        print(f"[Exploration] Skipping {skill} -- missing drawer, handle, or location")
                elif skill == "execute_sweep":
                    if obj and loc:
                        print(f"Calling {skill}({obj}, {loc})")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                    else:
                        print(f"[Exploration] Skipping {skill} -- missing object or location")
                elif skill == "execute_rotate":
                    # As there is no rotate in primitive_skills_static_lock_v2, skip or attempt if implemented
                    print("Skipping rotate as not in main domain")
                elif skill == "execute_go":
                    if loc and loc2:
                        print(f"Calling {skill}({loc}, {loc2})")
                        obs, reward, done = execute_go(env, task, loc, loc2)
                    else:
                        print(f"[Exploration] Skipping {skill} -- missing source or goal location")
                elif skill == "execute_gripper":
                    print(f"Calling {skill}()")
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f"[Exploration] Unknown skill: {skill}")
                
                # (Optional) Log any relevant output
                # print("Observation after action:", obs)
                # If reward or done signal indicates something interesting, print
                if 'reward' in locals():
                    print("Reward after action:", reward)
                if 'done' in locals() and done:
                    print("[Exploration] Done received after", skill)
                    break

            except Exception as e:
                print(f"[Exploration] Exception in skill {skill}: {str(e)}")
                # Here is where you might infer, based on precondition Violation, the missing predicate

        print("[Exploration] Finished all available skills. Check messages for failed predicates or errors hinting at missing predicates.")

        # === At this point, exploration is complete and you may use the info gathered to update your understanding of the domain ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
