# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine primitive skills.

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for robot-free predicate issues.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions is expected to be a dict: {'object_1': (x, y, z), ...}

        # -----------------------------------------------------------------------------
        # === EXPLORATION PHASE (diagnose missing predicates, e.g. robot-free) ========
        # -----------------------------------------------------------------------------
        #
        # Based on feedback, we need to check which predicate may be missing/undefined
        # or problematic. According to the domain, robot-free is a prerequisite for
        # many actions. If the task gets "stuck" due to robot-free not becoming true,
        # we will systematically try to restore this predicate with available skills.
        #
        # We use only available skills. Here, available skills include:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # We demonstrate restoring "robot-free" status using 'execute_gripper' when needed.

        def ensure_robot_free(env, task):
            '''
            Tries to ensure that the robot is in a "robot-free" state using available skills.
            Uses execute_gripper if available.

            Returns True if robot-free likely restored/completed, False otherwise.
            '''
            try:
                print("[Exploration] Attempting to free robot using 'execute_gripper' skill.")
                obs, reward, done = execute_gripper(env, task)
                return True
            except Exception as e:
                print(f"[Exploration] Could not restore robot-free: {e}")
                return False

        # Simulated high-level oracle plan.
        # The real plan should be derived from planning, but this place is where such a plan would be iterated.
        # For demonstration, we show detection of blocked actions and recovery using exploration phase.

        # For demonstration, we try a simple loop with some example objects (keys must exist in positions!)
        ball_name = None
        for obj_name in positions:
            if obj_name.startswith('ball') or obj_name.startswith('object'):
                ball_name = obj_name
                break

        # Also search for a drawer, if present
        drawer_name = None
        for obj_name in positions:
            if 'drawer' in obj_name:
                drawer_name = obj_name
                break

        # Example plan: Move to object, pick, move to drawer, open drawer, place object, close drawer

        success = False
        try:
            # 1. Move to object's location
            if ball_name:
                obj_pos = positions[ball_name]
                print(f"[Task] Moving to object {ball_name} at position {obj_pos}")
                obs, reward, done = execute_go(env, task, obj_pos)
            
            # 2. Ensure robot-free before pick
            robot_free_ok = ensure_robot_free(env, task)
            
            # 3. Pick the object
            if ball_name:
                print(f"[Task] Attempting to pick up {ball_name}")
                try:
                    obs, reward, done = execute_pick(env, task, ball_name, obj_pos)
                except Exception as e:
                    print(f"[Error] Failed to execute_pick: {e}")
                    robot_free_ok = ensure_robot_free(env, task)
                    # Try again
                    try:
                        obs, reward, done = execute_pick(env, task, ball_name, obj_pos)
                    except Exception as e2:
                        print(f"[Exploration] Still unable to pick {ball_name}: {e2}")

            # 4. Move to drawer's location (if available)
            if drawer_name:
                drawer_pos = positions[drawer_name]
                print(f"[Task] Moving to drawer {drawer_name} at position {drawer_pos}")
                obs, reward, done = execute_go(env, task, drawer_pos)

                # 5. Try to open drawer (pull)
                handle_name = None
                # Try to find handle-of relationship (not present in positions, assumed known if part of environment)
                # If not, use default object (e.g. handle) if present
                for key in positions:
                    if 'handle' in key:
                        handle_name = key
                        break
                if handle_name:
                    print(f"[Task] Attempting to pull open {drawer_name} using handle {handle_name}")
                    try:
                        obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_pos)
                    except Exception as e:
                        print(f"[Error] Failed to execute_pull: {e}")
                        robot_free_ok = ensure_robot_free(env, task)
                        try:
                            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_pos)
                        except Exception as e2:
                            print(f"[Exploration] Still unable to pull open {drawer_name}: {e2}")

                # 6. Place object into drawer
                if ball_name:
                    print(f"[Task] Placing {ball_name} into {drawer_name}")
                    try:
                        obs, reward, done = execute_place(env, task, ball_name, drawer_name, drawer_pos)
                    except Exception as e:
                        print(f"[Error] Failed to execute_place: {e}")
                        robot_free_ok = ensure_robot_free(env, task)
                        try:
                            obs, reward, done = execute_place(env, task, ball_name, drawer_name, drawer_pos)
                        except Exception as e2:
                            print(f"[Exploration] Still unable to place {ball_name} in {drawer_name}: {e2}")

                # 7. Close drawer
                print(f"[Task] Closing drawer {drawer_name}")
                try:
                    obs, reward, done = execute_push(env, task, drawer_name, drawer_pos)
                except Exception as e:
                    print(f"[Error] Failed to execute_push: {e}")
                    robot_free_ok = ensure_robot_free(env, task)
                    try:
                        obs, reward, done = execute_push(env, task, drawer_name, drawer_pos)
                    except Exception as e2:
                        print(f"[Exploration] Still unable to push {drawer_name} closed: {e2}")
            
            # Final status:
            print("[Task] Oracle-like plan execution finished.")
            success = True

        except Exception as e:
            print(f"[Task] Encountered unrecoverable error during plan execution: {e}")

        # Optionally, summary or status reporting here

        # -----------------------------------------------------------------------------
        # END OF EXPLORATION PHASE AND PLAN EXECUTION
        # -----------------------------------------------------------------------------

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()