# run_skeleton_task.py (Filled In)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # You may need to adjust these keys depending on your real object/drawer/location naming
        # Below are example names (replace with actual ones in your setup)
        robot_loc = positions.get('robot', None)  # Example: positions might have 'robot'
        drawer1_loc = positions.get('drawer1', None)
        handle1_loc = positions.get('handle1', None)
        obj_locs = {k: v for k, v in positions.items() if k.startswith('obj')}

        # Exploration step: detect missing predicate (e.g., 'drawer-unlocked drawer1')
        # For this, we use only the available skills, which are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # Since feedback suggests (drawer-unlocked drawer1) was missed,
        # try to interact with the drawer to discover its state before executing high-level actions.

        print("[Exploration] Probing for missing predicates before plan execution...")

        # Try to "pull" the drawer handle with available skills.
        # If 'execute_pull' fails, we infer the drawer may be locked.

        drawer_name = None
        handle_name = None
        for name in positions:
            if 'drawer' in name:
                drawer_name = name
            if 'handle' in name:
                handle_name = name
        if drawer_name is None or handle_name is None:
            print("[Exploration] Could not find drawer or handle in object positions!")
        else:
            try:
                # Step 1: Move to the location of the handle (assuming you must be at its location)
                current_location = positions.get('robot', positions.get('robot_initial', None))
                handle_location = positions.get(handle_name)
                if current_location and handle_location and current_location != handle_location:
                    print(f"[Exploration] Moving to handle location: {handle_location}")
                    obs, reward, done = execute_go(env, task, current_location, handle_location)
                else:
                    print("[Exploration] Robot already at handle location or locations unavailable.")

                # Step 2: Pick the handle (must have robot at handle's location)
                print(f"[Exploration] Attempting to pick handle: {handle_name}")
                obs, reward, done = execute_pick(env, task, handle_name, handle_location)

                # Step 3: Try to pull (open) the drawer with the handle, to test lock state
                print(f"[Exploration] Attempting to pull (open) the drawer {drawer_name} by handle {handle_name}...")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, handle_location)

                print("[Exploration] Drawer pull executed; if successful, drawer is unlocked. Otherwise, locked.")
            except Exception as e:
                print(f"[Exploration] Exception during handle pull: {e}")
                print("[Exploration] It is likely the drawer is locked or another precondition is missing.")

        # === Main Oracle Plan Execution ===
        # Here you would execute the actual planned sequence now that you checked the critical (missing) predicate.
        # Below is a minimal demonstration (replace with actual plan steps as needed).

        print("[Task] Executing main oracle plan...")

        try:
            # Suppose now: robot should pick up an object and place it in the drawer.

            # Example: Move to object location
            for obj_name, obj_location in obj_locs.items():
                # Step 1: Go to the location where the object is
                print(f"[Task] Moving to object {obj_name} at location {obj_location}.")
                obs, reward, done = execute_go(env, task, handle_location, obj_location)

                # Step 2: Pick the object
                print(f"[Task] Picking up object {obj_name} at location {obj_location}.")
                obs, reward, done = execute_pick(env, task, obj_name, obj_location)

                # Step 3: Move back to the drawer
                print(f"[Task] Moving to drawer {drawer_name} location {drawer1_loc}.")
                obs, reward, done = execute_go(env, task, obj_location, drawer1_loc)

                # Step 4: Place the object in the drawer, if drawer is open and not full
                print(f"[Task] Attempting to place object {obj_name} into drawer {drawer_name}.")
                try:
                    obs, reward, done = execute_place(env, task, obj_name, drawer_name, drawer1_loc)
                except Exception as place_e:
                    print(f"[Task] Could not place object {obj_name} in {drawer_name}: {place_e}")

                # For demonstration, only do one object
                break

        except Exception as main_e:
            print(f"[Task] Exception during oracle plan execution: {main_e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
