# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase to find missing predicate ---
        # According to the feedback: (robot-at drawer-area)
        # Try to move the robot to 'drawer-area' and see if this triggers expected effects
        try:
            print("[Exploration] Attempting to go to the drawer-area...")
            # You may need to map the robot's current location (from initial state or default) and 'drawer-area'
            # For this example, let's assume 'start-area' is the initial location (or replace accordingly)
            current_location = None
            drawer_area = 'drawer-area'
            # Derive the starting area from positions or the environment:
            # Heuristically grabbing the first location that is not 'drawer-area'
            for loc in positions:
                if loc != drawer_area and 'area' in loc:
                    current_location = loc
                    break
            if current_location is None:
                # Fallback to some default
                current_location = 'start-area'
            print(f"[Exploration] Moving from {current_location} --> {drawer_area}")

            # Use the available skill: execute_go
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_location,
                to_location=drawer_area
            )
            print("[Exploration] Executed 'execute_go'.")
        except Exception as e:
            print(f"[Exploration] Error during exploration move: {e}")

        # --- PLAN EXECUTION PHASE ---
        # At this point, the missing predicate (robot-at drawer-area) is now set.
        # Continue with the main sequence of oracle actions.
        #
        # You may define a plan as a list of steps, or hardcode for your scenario.
        # For demonstration, we show a typical manipulation sequence:
        try:
            # Example: pick up handle, pull to open the drawer, place an object inside, close the drawer.
            # Adjust names to match your environment and object_positions output.

            # 1. Move to drawer-area (if not already)
            # (already moved during exploration phase, so skip unless your plan requires further moves)

            # 2. Pick up the drawer handle (assuming handle name is known, e.g., 'handle_1' at drawer-area)
            handle_obj = None
            for name in positions:
                if 'handle' in name:
                    handle_obj = name
                    break
            if handle_obj is None:
                raise Exception("No handle object found in positions dictionary.")

            print(f"[Task] Picking handle: {handle_obj}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_obj,
                location=drawer_area
            )

            # 3. Pull the handle to open the drawer (assume drawer name e.g. 'drawer_1')
            drawer_obj = None
            for name in positions:
                if 'drawer' in name and name != drawer_area:
                    drawer_obj = name
                    break
            if drawer_obj is None:
                drawer_obj = 'drawer_1'  # Fallback

            print(f"[Task] Pulling handle to open: {drawer_obj} with {handle_obj}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_obj,
                handle_name=handle_obj,
                location=drawer_area
            )

            # 4. Pick some object to place in the drawer (e.g., 'object_1')
            obj_to_place = None
            for name in positions:
                if 'object' in name:
                    obj_to_place = name
                    break
            if obj_to_place is None:
                obj_to_place = 'object_1'  # Fallback

            print(f"[Task] Picking up object: {obj_to_place}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=obj_to_place,
                location=drawer_area
            )

            # 5. Place the object into the drawer (assume place requires: object, drawer, location)
            print(f"[Task] Placing {obj_to_place} into {drawer_obj}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=obj_to_place,
                drawer_name=drawer_obj,
                location=drawer_area
            )

            # 6. Push the drawer closed
            print(f"[Task] Pushing drawer {drawer_obj} closed.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_obj,
                location=drawer_area
            )

            print("[Task] Oracle plan executed.")
        except Exception as e:
            print(f"[Task] Error during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
