# run_skeleton_task.py (Completed and Specialized for Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use externally provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Skeleton for running predicate exploration in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Initialize and reset the task
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try possible facts, discover missing predicate ===
        #
        # Because the feedback indicates that planning failed possibly due to a missing predicate,
        # we perform exploration by systematically executing available predefined skills 
        # to see which predicates (effects/fluents/conditions) are missing or required for planning.
        #
        # The focus is on predicates that may be related to object identity, lock state, etc.
        #
        # The exploration domain (from the knowledge) suggests the following possibilities:
        #   identified, temperature-known, weight-known, durability-known, lock-known
        #
        # The available skills from domain:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        #  'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # Let's try to use each skill in safe order (where possible), catch failures, and log what predicates
        # are likely missing in the plan.

        # This exploration loop can be adapted based on actual objects and positions
        # For demonstration, assume we focus on the first object and first drawer we find.

        # Helper: Extract candidate objects/drawers/locations if available
        candidate_object = None
        candidate_location = None
        candidate_drawer = None
        candidate_handle = None
        candidate_handle_object = None

        # Try to extract any available object, drawer, location, handle from positions (generic logic)
        for name in positions:
            if "drawer" in name and candidate_drawer is None:
                candidate_drawer = name
            elif "handle" in name and candidate_handle is None:
                candidate_handle = name
            elif "object" in name and candidate_object is None:
                candidate_object = name
            elif ("loc" in name or "floor" in name) and candidate_location is None:
                candidate_location = name
            # Try to guess handles as objects too (since handle-of is object-drawer)
            elif "handle" in name and candidate_handle_object is None:
                candidate_handle_object = name

        # Backoff defaults
        if candidate_object is None and len(positions) > 0:
            for name in positions:
                if name != candidate_drawer and name != candidate_handle:
                    candidate_object = name
                    break
        if candidate_location is None and len(positions) > 0:
            candidate_location = list(positions.keys())[0]

        print("[Exploration] Candidates:", candidate_object, candidate_drawer, candidate_location, candidate_handle, candidate_handle_object)
        
        # For clarity, let us log all actions we will attempt
        exploration_skills = [
            ("execute_pick",   (candidate_object, candidate_location)),
            ("execute_place",  (candidate_object, candidate_drawer, candidate_location)),
            ("execute_push",   (candidate_drawer, candidate_location)),
            ("execute_pull",   (candidate_drawer, candidate_handle_object, candidate_location)),
            ("execute_sweep",  (candidate_object, candidate_location)),
            ("execute_rotate", ()),  # Not enough info; will skip if needed
            ("execute_go",     (candidate_location, candidate_location)),
            ("execute_gripper", ())
        ]
        
        # Try all skills, record which fail due to missing state/fluents, try to infer missing predicate
        for skill, args in exploration_skills:
            func = globals().get(skill, None)
            if func is None:
                print(f"[Exploration] Skill '{skill}' not implemented in skill_code; skipping.")
                continue
            try:
                print(f"[Exploration] Attempting skill: {skill} with args: {args}")
                if skill == "execute_pick":
                    if args[0] and args[1]:
                        obs, reward, done = func(env, task, obj=args[0], p=args[1])
                    else:
                        print("[Exploration] Missing parameters for execute_pick; skipping.")
                        continue
                elif skill == "execute_place":
                    if args[0] and args[1] and args[2]:
                        obs, reward, done = func(env, task, obj=args[0], d=args[1], p=args[2])
                    else:
                        print("[Exploration] Missing parameters for execute_place; skipping.")
                        continue
                elif skill == "execute_push":
                    if args[0] and args[1]:
                        obs, reward, done = func(env, task, d=args[0], p=args[1])
                    else:
                        print("[Exploration] Missing parameters for execute_push; skipping.")
                        continue
                elif skill == "execute_pull":
                    # pull needs a drawer, a handle-object, and location
                    if args[0] and args[1] and args[2]:
                        obs, reward, done = func(env, task, d=args[0], h=args[1], p=args[2])
                    else:
                        print("[Exploration] Missing parameters for execute_pull; skipping.")
                        continue
                elif skill == "execute_sweep":
                    if args[0] and args[1]:
                        obs, reward, done = func(env, task, o=args[0], p=args[1])
                    else:
                        print("[Exploration] Missing parameters for execute_sweep; skipping.")
                        continue
                elif skill == "execute_rotate":
                    obs, reward, done = func(env, task)
                elif skill == "execute_go":
                    if args[0] and args[1]:
                        obs, reward, done = func(env, task, from_loc=args[0], to_loc=args[1])
                    else:
                        print("[Exploration] Missing parameters for execute_go; skipping.")
                        continue
                elif skill == "execute_gripper":
                    obs, reward, done = func(env, task)
                else:
                    print(f"[Exploration] Unknown skill {skill}; skipping.")
                    continue

                print(f"[Exploration] Skill {skill} executed.")
                # Can add detail here: check obs, reward, etc.

            except Exception as e:
                print(f"[Exploration] Exception during {skill}: {str(e)}")
                print(f"[Exploration] Likely missing predicate or precondition for skill '{skill}'.")
                # Here you could do deeper inspection of exception to infer exactly what predicate is missing

        # === End Exploration Phase ===

        # After attempted exploration, developer can check console logs for which preconditions or predicates
        # must be added/updated in the PDDL domain to allow planning to proceed.
        print("===== Exploration phase complete. =====")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
