# run_skeleton_task.py (Complete Executable Code for Exploration & Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use primitives as provided

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions / State Information ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicates ===
        # Based on the feedback and the exploration domain, we need to actively explore unknown or missing object-level predicates, 
        # like "lock-known", "identified", "weight-known", etc. The predicate most relevant to drawer state is "lock-known".
        #
        # We will perform a sweep of exploration-relevant skills on objects/drawers, 
        # specifically focusing on detecting unknown lock states of drawers/handles. 
        # This mirrors the exploration domain's "execute_pull" for discovering lock status.

        # Available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll loop over relevant drawers and their handle objects, attempt pulls for lock-state exploration,
        # and include robust error handling.

        print("=== [Exploration] Beginning exploration to discover missing predicates... ===")
        # Placeholder: you need to adapt these names to your simulated environment's specific drawer/handle/object names!
        # Below is a generic logic.

        # 1. Get all drawers and handles from positions (assuming positions dict provides them)
        drawer_names = [name for name in positions if "drawer" in name.lower()]
        handle_names = [name for name in positions if "handle" in name.lower()]
        location_names = [name for name in positions if "location" in name.lower() or "site" in name.lower()]

        # Fallback: if locations are not specified, try using all unique position entries
        if not location_names:
            location_names = list(set([v['location'] for v in positions.values() if isinstance(v, dict) and 'location' in v]))

        # 2. Attempt to "pick" each handle (to simulate grasping for future "pull" to discover lock-known)
        for handle in handle_names:
            try:
                handle_info = positions[handle]
                # Inferring location for the handle
                if isinstance(handle_info, dict) and 'location' in handle_info:
                    handle_loc = handle_info['location']
                else:
                    handle_loc = None
                    # Try guessing location based on handle name, fallback: use first location
                    for loc in location_names:
                        if loc in handle:
                            handle_loc = loc
                            break
                    if not handle_loc and location_names:
                        handle_loc = location_names[0]

                if not handle_loc:
                    print(f"[Exploration] Can't determine location for handle {handle}, skipping.")
                    continue

                print(f"[Exploration] Attempting to pick handle [{handle}] at location [{handle_loc}] ...")
                # execute_pick(env, task, object_name, location_name)
                obs, reward, done = execute_pick(env, task, handle, handle_loc)
                print(f"[Exploration] Picked handle [{handle}]. Attempting to pull to sense lock...")

                # Find associated drawer (domain: (handle-of ?h - object ?d - drawer))
                # Try to infer mapping from handle to drawer
                drawer_for_handle = None
                for drawer in drawer_names:
                    if drawer.replace("drawer", "") in handle.replace("handle",""):
                        drawer_for_handle = drawer
                        break
                if not drawer_for_handle and drawer_names:
                    # Fallback: just select the first
                    drawer_for_handle = drawer_names[0]

                if drawer_for_handle:
                    print(f"[Exploration] Attempting execute_pull (drawer: {drawer_for_handle}, handle: {handle}, at: {handle_loc}) ...")
                    # execute_pull(env, task, drawer_name, handle_name, location_name)
                    try:
                        obs, reward, done = execute_pull(env, task, drawer_for_handle, handle, handle_loc)
                        print(f"[Exploration] Pulled handle [{handle}] for drawer [{drawer_for_handle}] at {handle_loc}.")
                    except Exception as e:
                        print(f"[Exploration] Failed to pull handle {handle} for drawer {drawer_for_handle}: {e}")
                else:
                    print(f"[Exploration] Cannot determine which drawer is connected to handle {handle}. Skipping pull.")

            except Exception as e:
                print(f"[Exploration] Error during pick/pull for handle {handle}: {e}")

        print("=== [Exploration] Done. If a missing predicate (e.g. lock-known) exists, it has now been triggered in the environment! ===")

        # === Main Task Plan Execution Phase (if needed) ===
        # After exploration you may continue with the main plan as per the oracle plan, e.g. picking, placing, moving, etc.
        #
        # Example (disabled, as details will depend on your specific task/plan):
        #
        # for obj in relevant_objects:
        #     obs, reward, done = execute_pick(env, task, obj, location)
        #     if done:
        #         print("[Main Task] Finished after first pick!")
        #         break
        #
        # Implement main oracle plan here as needed.

        # TODO: Complete oracle plan execution here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()