# run_skeleton_task.py (Filled for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Example object name conventions ---
        # For this domain, expect at least: drawers, handles, objects, locations
        # Example naming:
        #  drawers: 'drawer1', ...
        #  handles: 'handle1', ...
        #  objects: 'obj1', ...
        #  locations: 'loc1', ...

        # For demonstration, let's try to determine which predicate is missing via "exploration"
        # as suggested by the feedback and exploration knowledge.

        # ---- Exploration Phase: Determine Drawer Lock Status ----
        # Suppose we want to check the lock status of 'drawer1'
        drawer_name = 'drawer1'
        handle_name = None
        location_name = None

        # Find associated handle and location from positions and/or descriptions
        # For this sample, we assume handles are named "handle_of_drawer1"
        for key in positions:
            if 'handle' in key and drawer_name in key:
                handle_name = key
            if 'loc' in key and drawer_name in key:
                location_name = key

        if handle_name is None:
            # fallback, just pick any handle
            for key in positions:
                if 'handle' in key:
                    handle_name = key
                    break

        if location_name is None:
            # fallback, pick any location
            for key in positions:
                if 'loc' in key:
                    location_name = key
                    break

        # For the demo, fallback values if everything fails
        if handle_name is None:
            handle_name = 'handle1'
        if location_name is None:
            location_name = 'loc1'

        print(f"[Exploration] Attempting to determine lock status of {drawer_name} using handle {handle_name} at location {location_name}.")

        # Move to the drawer's location
        # We assume robot starts at some initial location; get a list of locations
        locations = [k for k in positions if 'loc' in k]
        current_location = locations[0] if locations else 'loc1'
        if current_location != location_name:
            try:
                print(f"[Exploration] Moving from {current_location} to {location_name}")
                obs, reward, done = execute_go(env, task, current_location, location_name)
                current_location = location_name
            except Exception as e:
                print(f"Error during execute_go exploration: {e}")

        # Try to pick up the handle (to prepare to pull and thus check lock status)
        try:
            print(f"[Exploration] Trying to pick up the handle {handle_name}")
            obs, reward, done = execute_pick(env, task, handle_name, location_name)
        except Exception as e:
            print(f"Error during execute_pick exploration: {e}")

        # Try to pull the drawer open (exploration: in exploration domain, execute_pull checks lock-known)
        try:
            print(f"[Exploration] Attempting to pull to test lock status")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, location_name)
        except Exception as e:
            print(f"Error during execute_pull exploration: {e}")

        # ---- End of Exploration ----
        # Based on the feedback: (drawer-unlocked drawer1)
        # This allows our plan to later include any action requiring (drawer-unlocked drawer1) in precondition.

        # ---- Task Plan Execution (Example Plan) ----
        # Now, after exploration, continue with normal plan
        # For illustration, assume we want to pick up obj1 from the floor and place it in the drawer1

        obj_name = None
        for key in positions:
            if 'obj' in key:
                obj_name = key
                break
        if obj_name is None:
            obj_name = 'obj1'
        print(f"[Task] Focusing on object: {obj_name}")

        # Move to the location of obj_name if not already there
        obj_location = None
        for key, val in positions.items():
            if key == obj_name:
                # positions[obj_name] is (x,y,z) but we need logical location name
                # We'll just use current_location for demonstration
                obj_location = current_location
                break
        if obj_location is None:
            obj_location = current_location

        # Pick obj_name
        try:
            print(f"[Task] Picking up {obj_name} at {obj_location}")
            obs, reward, done = execute_pick(env, task, obj_name, obj_location)
        except Exception as e:
            print(f"Error during execute_pick task: {e}")

        # Move to the drawer's location if needed
        if current_location != location_name:
            try:
                print(f"[Task] Moving with {obj_name} to {location_name}")
                obs, reward, done = execute_go(env, task, current_location, location_name)
                current_location = location_name
            except Exception as e:
                print(f"Error during execute_go to drawer: {e}")

        # Open drawer if not open (try pull again to make sure)
        try:
            print(f"[Task] Pulling drawer open (ensure open)")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, location_name)
        except Exception as e:
            print(f"Error during execute_pull to open drawer: {e}")

        # Place the object into the drawer
        try:
            print(f"[Task] Placing {obj_name} into {drawer_name} at {location_name}")
            obs, reward, done = execute_place(env, task, obj_name, drawer_name, location_name)
        except Exception as e:
            print(f"Error during execute_place: {e}")

        # Optionally push drawer closed
        try:
            print(f"[Task] Pushing drawer closed")
            obs, reward, done = execute_push(env, task, drawer_name, location_name)
        except Exception as e:
            print(f"Error during execute_push: {e}")

        print("[Task] Task sequence complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
