# run_skeleton_task.py (Completed with Exploration to Identify Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Only use predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Probe for Missing Predicate ===
        # There was a previous plan timeout, likely due to a missing predicate (such as lock-known, durability-known, temperature-known, etc.)
        # Based on the exploration domain and available skills, we need to use available skills in an exploratory sequence to identify possible missing predicates before running a full plan.

        # Note: This is exploratory pseudo-logic; you should adjust object/location names if your environment uses different ones.

        # Assume you want to explore over all objects and at all locations to determine if some "knowledge predicates" are present or need to be triggered.
        # Available exploration knowledge predicates: identified, temperature-known, weight-known, durability-known, lock-known.
        # But only call the provided skills.

        # Find all object/location keys from positions dict (best effort)
        object_keys = [k for k, v in positions.items() if 'object' in k or 'ball' in k or 'item' in k or 'cup' in k]
        location_keys = [k for k, v in positions.items() if 'room' in k or 'drawer' in k or 'table' in k or 'location' in k]

        # Also try to infer a handle or drawer handle key if available
        handle_keys = [k for k in positions.keys() if 'handle' in k]
        drawer_keys = [k for k in positions.keys() if 'drawer' in k]

        # Assume your robot starts free and empty-handed, as in the initial state

        # Try to sweep/fake explore over all objects-on-floor to check for knowledge predicates
        for obj in object_keys:
            try:
                print(f"[Exploration] Sweeping or gripper check for object: {obj}")
                # execute_sweep is effect-less but could be used for sensory exploration
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    positions[obj]
                )
            except Exception as e:
                print(f"[WARN] Sweep exploration failed for {obj}: {e}")
            try:
                print(f"[Exploration] Gripper check (could be redundant)...")
                obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print(f"[WARN] Gripper exploration failed: {e}")

        # Try to use go or move actions to probe also per location
        for loc in location_keys:
            try:
                print(f"[Exploration] Moving robot to {loc} for identification...")
                current_locations = [k for k in location_keys if env.robot_is_at(k)]
                if current_locations:
                    from_loc = current_locations[0]
                else:
                    from_loc = loc  # fallback (should improve with better state)

                if from_loc != loc:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_loc,
                        loc
                    )
            except Exception as e:
                print(f"[WARN] Move exploration to {loc} failed: {e}")

        # If drawer/handle present, try to pull/push to check lock-known
        for d in drawer_keys:
            for h in handle_keys:
                try:
                    print(f"[Exploration] Testing drawer pull ({d}) with handle ({h})...")
                    # To execute_pull we must be holding the handle, at right location, with drawer locked/unlocked, etc.
                    # First try picking up handle
                    try:
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            h,
                            positions[d]  # assuming handle is near/at drawer
                        )
                    except Exception as e:
                        print(f"[WARN] Could not pick handle {h} for drawer {d}: {e}")

                    # Now try pulling
                    try:
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            d,
                            h,
                            positions[d]
                        )
                    except Exception as e:
                        print(f"[WARN] Could not pull drawer {d} with handle {h}: {e}")

                except Exception as e:
                    print(f"[WARN] Exploration pull/push for {d} failed: {e}")

        print("[Exploration] Finished all knowledge predicate exploration.")

        # === Main Task Plan Execution (replace this with actual oracle plan as needed) ===

        # Example high-level plan for oracle (project provided plan should go here)
        # For instance, for each object, pick and place into a drawer:
        for obj in object_keys:
            for d in drawer_keys:
                try:
                    # Move robot to drawer location if not already there
                    current_locations = [k for k in location_keys if env.robot_is_at(k)]
                    if not current_locations or current_locations[0] != d:
                        obs, reward, done = execute_go(
                            env,
                            task,
                            current_locations[0] if current_locations else d,
                            d
                        )
                    # Pick the object if on floor in drawer location
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj,
                        positions[d]
                    )
                    # If the drawer is closed, open it (pick up handle and pull)
                    h_candidates = [h for h in handle_keys if h in positions]
                    h = h_candidates[0] if h_candidates else None
                    if h:
                        # Pick handle
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            h,
                            positions[d]
                        )
                        # Pull
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            d,
                            h,
                            positions[d]
                        )
                    # Place object in drawer
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        d,
                        positions[d]
                    )
                    # Optionally, push (close the drawer)
                    obs, reward, done = execute_push(
                        env,
                        task,
                        d,
                        positions[d]
                    )
                except Exception as e:
                    print(f"[Task-Plan] Failed action sequence for {obj} and {d}: {e}")

        # If the task ends before full plan (done flag), break
        print("[Task] Main oracle plan finished.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()