# run_skeleton_task.py (Fully Completed for Exploration Phase)
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing primitives

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Exploration-focused skeleton for identifying missing predicates via exploration skills.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset task and get initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # --- Exploration Knowledge: Purpose ---
        # The fast-downward planner timed out (likely due to a missing predicate or knowledge, such as lock or weight).
        # So, we systematically explore using only skills: execute_pick, execute_place, execute_push, execute_pull,
        # execute_sweep, execute_rotate, execute_go, execute_gripper, to find clues in the environment.

        # === Retrieve Object Positions (names and positions) ===
        positions = get_object_positions()

        # Attempt to infer types of available objects, drawers, locations, handles, etc.
        # This is highly environment-dependent but for exploration we try each skill on reasonable guesses.

        # Try to automatically find main types of objects (very basic heuristic)
        all_names = list(positions.keys())
        possible_drawers = [n for n in all_names if 'drawer' in n.lower()]
        possible_handles = [n for n in all_names if 'handle' in n.lower()]
        possible_objects = [n for n in all_names if n not in possible_drawers and n not in possible_handles]
        possible_locations = []
        # Try to extract unique set of locations from object position tuples if possible
        for name in all_names:
            pos = positions[name]
            if isinstance(pos, tuple) and len(pos) >= 2:
                possible_locations.append(name)
        robot_location = None
        # Fallback: any object named 'robot' or similar
        for n in all_names:
            if 'robot' in n.lower():
                robot_location = n
                break

        # If environment encodes location names, else default
        if len(possible_locations) == 0:
            possible_locations = ['location1', 'location2']

        # 1. Try "execute_go" between possible locations to check robot mobility
        print("[Exploration] Testing execute_go between likely locations...")
        for i in range(len(possible_locations)):
            for j in range(len(possible_locations)):
                if i != j:
                    from_loc = possible_locations[i]
                    to_loc = possible_locations[j]
                    try:
                        print(f" - execute_go: {from_loc} → {to_loc}")
                        # Skill may require different params; passing only locations as in domain
                        obs, reward, done, _ = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print("   [Info] Reached destination or ended.")
                            break
                    except Exception as e:
                        print(f"   [Warning] execute_go failed from {from_loc} to {to_loc}: {e}")

        # 2. Try to pick up every object from every location (execute_pick)
        print("[Exploration] Testing execute_pick for every object...")
        for obj_name in possible_objects:
            for loc in possible_locations:
                try:
                    print(f" - execute_pick: {obj_name} @ {loc}")
                    obs, reward, done, _ = execute_pick(env, task, obj_name, loc)
                    if done:
                        print("   [Info] Picked or task ended for", obj_name)
                        break
                except Exception as e:
                    print(f"   [Warning] execute_pick failed for {obj_name} @ {loc}: {e}")

        # 3. Try execute_place on drawers for any object now possibly held
        print("[Exploration] Testing execute_place for all relevant (object, drawer, location) triples...")
        for obj_name in possible_objects:
            for drawer_name in possible_drawers:
                for loc in possible_locations:
                    try:
                        print(f" - execute_place: {obj_name} into {drawer_name} @ {loc}")
                        obs, reward, done, _ = execute_place(env, task, obj_name, drawer_name, loc)
                        if done:
                            print(f"   [Info] Placed {obj_name} in {drawer_name} or task ended")
                            break
                    except Exception as e:
                        print(f"   [Warning] execute_place failed for {obj_name} in {drawer_name} @ {loc}: {e}")

        # 4. Try opening drawers using handles with execute_pull
        print("[Exploration] Testing execute_pull (handle/drawer/location)...")
        for drawer_name in possible_drawers:
            for handle_name in possible_handles:
                for loc in possible_locations:
                    try:
                        print(f" - execute_pull: {drawer_name}, {handle_name}, {loc}")
                        obs, reward, done, _ = execute_pull(env, task, drawer_name, handle_name, loc)
                        if done:
                            print("   [Info] Drawer possibly pulled open or task ended")
                            break
                    except Exception as e:
                        print(f"   [Warning] execute_pull failed for {drawer_name}, {handle_name}, {loc}: {e}")

        # 5. Try pushing drawers (to close them)
        print("[Exploration] Testing execute_push for drawers at locations...")
        for drawer_name in possible_drawers:
            for loc in possible_locations:
                try:
                    print(f" - execute_push: {drawer_name} @ {loc}")
                    obs, reward, done, _ = execute_push(env, task, drawer_name, loc)
                    if done:
                        print("   [Info] Drawer closed or task ended")
                        break
                except Exception as e:
                    print(f"   [Warning] execute_push failed for {drawer_name} @ {loc}: {e}")

        # 6. Try executing execute_sweep on all objects and locations
        print("[Exploration] Testing execute_sweep for all objects...")
        for obj_name in possible_objects:
            for loc in possible_locations:
                try:
                    print(f" - execute_sweep: {obj_name} @ {loc}")
                    obs, reward, done, _ = execute_sweep(env, task, obj_name, loc)
                    if done:
                        print("   [Info] Swept or task ended for", obj_name)
                        break
                except Exception as e:
                    print(f"   [Warning] execute_sweep failed for {obj_name} @ {loc}: {e}")

        # 7. Try execute_gripper (no parameters)
        print("[Exploration] Testing execute_gripper (gripper open/close)...")
        try:
            obs, reward, done, _ = execute_gripper(env, task)
            print("   [Info] Gripper tested")
        except Exception as e:
            print(f"   [Warning] execute_gripper failed: {e}")

        print("[Exploration] Complete. Review warnings and domain feedback to identify missing predicates.")

        # Note: The missing predicate is likely to relate to lock or knowledge gained via pull/open, as indicated in feedback.
        # The warning and printouts will help diagnose "missing knowledge" based on what actions succeed or fail.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()