# run_skeleton_task.py (Completed Version)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use pre-defined skills ONLY

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''

    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # The actual positions dictionary should have physical objects and handles as keys.
        positions = get_object_positions()

        # ---------------------- EXPLORATION PHASE ------------------------
        # Per feedback (holding handle_bottom) and to discover missing predicate (likely lock-known or similar),
        # perform an explicit exploration to identify relevant object properties.

        # Example: Identify all candidate handles on drawers
        drawer_handles = []
        drawers = []
        for name in positions:
            if "handle" in name:
                drawer_handles.append(name)
            if "drawer" in name:
                drawers.append(name)

        # Suppose we have a main handle (handle_bottom) and drawer (drawer_main), adjust if object names differ
        handle_name = None
        drawer_name = None
        for h in drawer_handles:
            if "bottom" in h:
                handle_name = h
                break
        if not handle_name and drawer_handles:
            handle_name = drawer_handles[0]
        for d in drawers:
            if "main" in d:
                drawer_name = d
                break
        if not drawer_name and drawers:
            drawer_name = drawers[0]

        # Also locate initial robot position and candidate locations
        robot_location = None
        candidate_locations = []
        for name in positions:
            if "robot" in name or "init" in name:
                robot_location = name
            elif "loc" in name or "room" in name or "table" in name or "floor" in name:
                candidate_locations.append(name)
        # Fallback if not found
        if not robot_location:
            if candidate_locations:
                robot_location = candidate_locations[0]
            else:
                robot_location = "location_1"  # Default fallback

        # For demo: list all locations (to be more robust in unknown scenes)
        all_locations = []
        for name in positions:
            if "loc" in name or "room" in name or "table" in name or "floor" in name:
                all_locations.append(name)
        if not all_locations:
            all_locations = list(positions.keys())

        print(f"[Exploration] Robot initially at: {robot_location}")
        print(f"[Exploration] Drawer: {drawer_name}, Handle: {handle_name}")

        # ---- Attempt pick up on the handle (simulate 'holding handle_bottom') ----
        try:
            handle_pos = positions[handle_name]
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=handle_name,
                location=robot_location,
                max_steps=100
            )
            print(f"[Exploration] Picked up handle '{handle_name}'.")
        except Exception as e:
            print(f"[Exploration] Failed to pick up handle: {e}")

        # ---- Exploration: Try pulling the handle to discover missing predicate (lock-known) ----
        # Per the exploration knowledge, execute_pull will let us know if the lock status is discoverable.
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                handle=handle_name,
                location=robot_location,
                max_steps=100
            )
            print(f"[Exploration] Pulled handle '{handle_name}' to test for lock status (lock-known).")
        except Exception as e:
            print(f"[Exploration] Failed to pull handle (possible locked state or missing lock-known predicate): {e}")

        # At this point, PDDL feedback and the effect of pulling should tell us which predicate was missing.
        print("[Exploration] Based on the action, check if 'lock-known' or related predicates are triggered in the environment.")

        # --------------------- TASK EXECUTION PHASE ----------------------

        # Proceed to main planning (example: open the drawer, place an object, and close it)
        # 1. Move robot to position (if required - insert logic if needed)
        # 2. Pick up an object from the floor
        # 3. Open drawer (must be unlocked and holding its handle)
        # 4. Place object in drawer
        # 5. Close the drawer

        # Simple example with available skill names:
        # Find an object that's "on-floor"
        floor_objects = [name for name in positions if "obj" in name or "ball" in name or "item" in name]
        obj_name = floor_objects[0] if floor_objects else None

        # Step 1: Pick up the object from the floor
        if obj_name:
            try:
                obj_pos = positions[obj_name]
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=obj_name,
                    location=robot_location,
                    max_steps=100
                )
                print(f"[Task] Picked up object '{obj_name}'.")
            except Exception as e:
                print(f"[Task] Failed to pick up object '{obj_name}': {e}")

        # Step 2: If needed, re-pick the handle to open the drawer
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=handle_name,
                location=robot_location,
                max_steps=100
            )
            print(f"[Task] (Re)Picked handle '{handle_name}' to open the drawer.")
        except Exception as e:
            print(f"[Task] Could not pick up handle again: {e}")

        # Step 3: Pull the handle to open the drawer
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                handle=handle_name,
                location=robot_location,
                max_steps=100
            )
            print(f"[Task] Pulled handle '{handle_name}' to open drawer '{drawer_name}'.")
        except Exception as e:
            print(f"[Task] Failed to open drawer: {e}")

        # Step 4: Place the object in the drawer (if still holding, as may be needed)
        # Simulate place (only if object pick action succeeded)
        if obj_name:
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj_name=obj_name,
                    drawer=drawer_name,
                    location=robot_location,
                    max_steps=100
                )
                print(f"[Task] Placed object '{obj_name}' in drawer '{drawer_name}'.")
            except Exception as e:
                print(f"[Task] Failed to place object '{obj_name}' in drawer '{drawer_name}': {e}")

        # Step 5: Push the drawer to close it
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer=drawer_name,
                location=robot_location,
                max_steps=100
            )
            print(f"[Task] Closed drawer '{drawer_name}'.")
        except Exception as e:
            print(f"[Task] Failed to close drawer: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
