# run_skeleton_task.py (Executable, Guided by Exploration for Missing Predicates)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use pre-implemented skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions (example: returns {'drawer1':..., 'objectA':..., ...}) ===
        positions = get_object_positions()

        # --- EXPLORATION PHASE ---
        # Feedback indicates there is a missing predicate; we must explore to discover it.
        # This phase uses only known skills, e.g., execute_go, execute_pick, execute_pull, etc.

        # Suppose we have the names of drawers, handles, and objects from positions:
        drawers = [name for name in positions.keys() if 'drawer' in name.lower()]
        handles = [name for name in positions.keys() if 'handle' in name.lower()]
        objects = [name for name in positions.keys() if 'object' in name.lower() or 'item' in name.lower()]
        locations = [name for name in positions.keys() if 'location' in name.lower() or 'room' in name.lower()]
        robot_location = None
        # Try to get the robot's starting location (if provided by observations or positions)
        for k in positions.keys():
            if 'robot' in k.lower():
                robot_location = k
                break

        if robot_location is None and len(locations) > 0:
            robot_location = locations[0]

        # For demonstration, assume 1 drawer, 1 handle, 1 object, 2 locations at minimum
        if not drawers or not handles or not objects or len(positions) < 2:
            print("[Exploration] Not enough information to start exploration; exiting.")
            return

        # Pick representative items
        drawer = drawers[0]
        handle = handles[0]
        obj = objects[0]
        locs = [k for k in positions.keys() if k != robot_location]
        if len(locs) < 1:
            print("[Exploration] Not enough locations for exploration.")
            return
        target_location = locs[0]

        # 1. Move to the object's location if not there already
        try:
            if robot_location is not None and target_location is not None and robot_location != target_location:
                print(f"[Exploration] Moving robot from {robot_location} to {target_location}")
                obs, reward, done, info = execute_go(env, task, robot_location, target_location)
                robot_location = target_location
        except Exception as e:
            print("[Exploration] execute_go failed:", e)

        # 2. Attempt to pick the object from the floor (to check predicates involving 'on-floor', 'hand-empty', etc.)
        try:
            print(f"[Exploration] Attempting to pick up object '{obj}' at location '{robot_location}'")
            obs, reward, done, info = execute_pick(env, task, obj, robot_location)
        except Exception as e:
            print("[Exploration] execute_pick failed (maybe not on floor or hand not free):", e)

        # 3. Attempt to open the drawer, which requires the handle and unlock status
        try:
            print(f"[Exploration] Attempting to pull handle '{handle}' on drawer '{drawer}' at location '{robot_location}'")
            obs, reward, done, info = execute_pick(env, task, handle, robot_location)
            obs, reward, done, info = execute_pull(env, task, drawer, handle, robot_location)
        except Exception as e:
            print("[Exploration] execute_pull failed (maybe drawer locked or handle status unknown):", e)

        # 4. Attempt to place the object in the drawer
        try:
            print(f"[Exploration] Attempting to place object '{obj}' in drawer '{drawer}' at location '{robot_location}'")
            obs, reward, done, info = execute_place(env, task, obj, drawer, robot_location)
        except Exception as e:
            print("[Exploration] execute_place failed (maybe drawer not open, or full, or not holding):", e)

        # 5. Attempt to push/close the drawer
        try:
            print(f"[Exploration] Attempting to close drawer '{drawer}' at location '{robot_location}'")
            obs, reward, done, info = execute_push(env, task, drawer, robot_location)
        except Exception as e:
            print("[Exploration] execute_push failed (maybe drawer not open):", e)

        # 6. Additional exploration: Try all the skills with reasonable arguments to trigger missing predicate errors
        # This may expose any missing predicates (e.g. lock-known, temperature-known, etc.)
        for skill_name in [
            "execute_sweep", "execute_gripper", "execute_go", "execute_pick", "execute_place", "execute_pull", "execute_push"
        ]:
            try:
                print(f"[Exploration] Testing skill: {skill_name}")
                func = globals().get(skill_name)
                # Call with various plausible arg combinations
                if skill_name == "execute_sweep":
                    obs, reward, done, info = func(env, task, obj, robot_location)
                elif skill_name == "execute_gripper":
                    obs, reward, done, info = func(env, task)
                elif skill_name == "execute_go":
                    for loc in locs:
                        if loc != robot_location:
                            obs, reward, done, info = func(env, task, robot_location, loc)
                            robot_location = loc
                            break
                elif skill_name == "execute_pick":
                    obs, reward, done, info = func(env, task, obj, robot_location)
                elif skill_name == "execute_place":
                    obs, reward, done, info = func(env, task, obj, drawer, robot_location)
                elif skill_name == "execute_pull":
                    obs, reward, done, info = func(env, task, drawer, handle, robot_location)
                elif skill_name == "execute_push":
                    obs, reward, done, info = func(env, task, drawer, robot_location)
            except Exception as e:
                print(f"[Exploration] {skill_name} failed:", e)

        print("===== Exploration complete. Check above for missing predicate errors or abnormal feedback. =====")

        # === NORMAL TASK LOGIC: Insert Oracle Plan execution here ===
        # (Your oracle plan should be executed after you are sure about the missing predicates & required actions.)

        # EXAMPLE: Insert your sequential plan here using only the available skill calls,
        # using the information gathered and any required arguments.
        # For instance:
        # obs, reward, done, info = execute_go(env, task, ...)
        # obs, reward, done, info = execute_pick(env, task, ...)
        # obs, reward, done, info = execute_pull(env, task, ...)
        # obs, reward, done, info = execute_place(env, task, ...)
        # Check for 'done' and manage reward/status accordingly.

        # TODO: Replace below with your actual planned sequence once the predicates are found

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()