# run_skeleton_task.py (Filled Executable with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Predefined skills: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Useful Handles ===
        positions = get_object_positions()
        # Because this is generic, we'll attempt to detect the objects required for the exploration phase.
        # The feedback mentions: (robot-at handle_pose)
        # Therefore, we need to infer what is "handle_pose". Let's look for a likely handle.

        # Try to recover commonly used keys for objects, drawers, handles, and locations
        handle_pose = positions.get('handle_pose', None)
        drawer_pose = positions.get('drawer_pose', None)
        robot_pose = positions.get('robot_pose', None)
        # If these are missing, try a heuristic guess from available keys
        if handle_pose is None:
            # Try to get first object that includes 'handle' in its name
            for k in positions.keys():
                if 'handle' in k:
                    handle_pose = positions[k]
                    break
        if drawer_pose is None:
            for k in positions.keys():
                if 'drawer' in k:
                    drawer_pose = positions[k]
                    break
        if robot_pose is None:
            for k in positions.keys():
                if 'robot' in k:
                    robot_pose = positions[k]
                    break

        # For generality, we also collect all objects and locations
        all_objects = [k for k in positions.keys() if 'object' in k]
        all_locations = [k for k in positions.keys() if 'pose' in k or 'location' in k]

        # ---------- EXPLORATION PHASE ----------
        # This phase is to investigate missing predicate or knowledge regarding feedback
        # The feedback is: (robot-at handle_pose)
        print("[Exploration] Begin exploration phase to check for missing predicate knowledge (robot-at handle_pose)")

        # Assumes available skills: execute_go, execute_pick, execute_pull, etc.
        # We check if robot can move to handle_pose and identify any errors
        from_location = robot_pose if robot_pose is not None else (0, 0, 0)
        to_location = handle_pose if handle_pose is not None else (0, 0, 0)

        # Attempt to move the robot to the handle pose/location
        try:
            print(f"[Exploration] Attempting execute_go from {from_location} to {to_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            print(f"[Exploration] Robot position after move: {to_location}")
        except Exception as e:
            print(f"[Exploration] Error while executing execute_go to handle_pose: {e}")

        # Assess if robot is at handle_pose, and try to interact (e.g., pick handle)
        handle_obj_name = None
        for name in positions.keys():
            if 'handle' in name:
                handle_obj_name = name
                break
        if handle_obj_name:
            handle_pos = positions[handle_obj_name]
            # Try to pick (simulate a grip) the handle
            try:
                print(f"[Exploration] Attempting execute_pick for handle object '{handle_obj_name}' at {handle_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle_obj_name,
                    to_location
                )
                print(f"[Exploration] Picked handle object: {handle_obj_name}")
            except Exception as e:
                print(f"[Exploration] Error while executing execute_pick on handle object: {e}")

            # Try to execute_pull (pull the drawer using handle)
            try:
                print(f"[Exploration] Attempting execute_pull with handle '{handle_obj_name}' at {to_location}")
                drawer_name = None
                for k in positions.keys():
                    if 'drawer' in k:
                        drawer_name = k
                        break
                if drawer_name:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name,
                        handle_obj_name,
                        to_location
                    )
                    print(f"[Exploration] Pulled drawer '{drawer_name}' with handle '{handle_obj_name}'")
                else:
                    print("[Exploration] No drawer object found in positions for execute_pull.")
            except Exception as e:
                print(f"[Exploration] Error while executing execute_pull: {e}")
        else:
            print("[Exploration] No handle object found in positions for pick/pull actions.")

        print("[Exploration] Exploration phase completed. Check logs to diagnose missing predicate issues.")

        # === Example Oracle Plan Execution (Stub; fill as per your plan) ===
        # For demonstration, include a representative plan using only available skills and detected objects
        # (This could be replaced by plan parsed from an external source)
        plan = []
        # Example: Move to handle, pick the handle, pull the drawer
        if robot_pose and handle_pose:
            plan.append(('execute_go', robot_pose, handle_pose))
        if handle_obj_name and handle_pose:
            plan.append(('execute_pick', handle_obj_name, handle_pose))
        if drawer_pose and handle_obj_name and handle_pose:
            plan.append(('execute_pull', drawer_pose, handle_obj_name, handle_pose))
        # You can further append place, push, etc. as needed

        # Execute the oracle plan step-by-step
        for step in plan:
            action = step[0]
            print(f"[Plan] Executing action: {action} with params {step[1:]}")
            try:
                if action == 'execute_go':
                    obs, reward, done = execute_go(env, task, step[1], step[2])
                elif action == 'execute_pick':
                    obs, reward, done = execute_pick(env, task, step[1], step[2])
                elif action == 'execute_pull':
                    obs, reward, done = execute_pull(env, task, step[1], step[2], step[3])
                elif action == 'execute_place':
                    obs, reward, done = execute_place(env, task, step[1], step[2], step[3])
                elif action == 'execute_push':
                    obs, reward, done = execute_push(env, task, step[1], step[2])
                # Add branches for other available skills if needed
                else:
                    print(f"[Plan] Unsupported action in plan: {action}")
                # Stop if task is done
                if done:
                    print("[Plan] Task ended successfully.")
                    break
            except Exception as e:
                print(f"[Plan] Exception occurred during {action}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()