# run_skeleton_task.py (Executable Version for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills as-is

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running the exploration phase in your simulation.
    This implements exploration to help discover missing predicates (such as "drawer-unlocked drawer2").
    '''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # The actual keys depend on the scenario/environment
        # Example keys (may vary): 'drawer2', 'handle2', 'robot', 'loc1', 'loc2', etc.

        # Print available positions for debugging
        print("Available positions:", positions.keys())

        # Example object and location assignments
        # These must reference actual keys in your simulation setup.
        # For demonstrative purposes, we assume:
        #  - 'drawer2' is the object whose predicate (drawer-unlocked drawer2) is in question
        #  - 'handle2' is the handle object attached to drawer2
        #  - 'robot_base' is the robot's current/base position
        #  - 'drawer2_location' is the location of drawer2

        # Please adapt keys below as needed for your scenario.
        drawer_object = 'drawer2'
        handle_object = 'handle2'
        robot_position = positions.get('robot_base', None)
        drawer_location = positions.get('drawer2_location', None)
        handle_position = positions.get('handle2_position', None)

        # Ensure required objects/positions exist
        if drawer_object not in positions:
            print(f"Error: '{drawer_object}' object position unavailable in get_object_positions(). Aborting exploration.")
            return
        if handle_object not in positions:
            print(f"Error: '{handle_object}' handle position unavailable in get_object_positions(). Aborting exploration.")
            return

        # Determine start location for robot if needed – this may depend on your env
        start_location = None
        for k in positions:
            if 'robot' in k and 'base' in k:
                start_location = positions[k]
        if start_location is None:
            # Fallback: pick any location
            for k in positions:
                if 'location' in k or 'loc' in k:
                    start_location = positions[k]
                    break
        if start_location is None:
            print("Error: Could not determine a start location for the robot.")
            return

        # === Exploration Phase ===
        print("[Exploration] Starting predicate exploration for drawer lock state.")

        # Move to the region near the drawer (if not already there)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=start_location,
                to_location=positions[drawer_object],  # This should be a key with position of drawer2
                max_steps=80,
                approach_distance=0.20
            )
            print(f"[Exploration] Robot moved to drawer2: {positions[drawer_object]}")
        except Exception as e:
            print(f"Error while moving to drawer2: {str(e)}")
            return

        # Pick the handle in order to pull (optional: can identify before picking if needed)
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_id=handle_object,
                object_position=positions[handle_object],
                approach_distance=0.15,
                max_steps=80,
                threshold=0.01
            )
            print(f"[Exploration] Picked up handle2 to check lock status.")
        except Exception as e:
            print(f"Error during picking handle2: {str(e)}")
            return

        # Try pulling the drawer using execute_pull (this action may reveal lock status)
        # execute_pull requires (drawer, handle, location)
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_object,
                handle_name=handle_object,
                location=positions[drawer_object],
                max_steps=80
            )
            print(f"[Exploration] Attempted to pull drawer2 using handle2 to check lock state.")
        except Exception as e:
            print(f"[Exploration] Pull failed; drawer might be locked. Exception: {str(e)}")

            print("[Exploration] Hypothesis: Missing predicate is 'drawer-unlocked drawer2' (drawer2 is locked).")
            # This is the missing predicate suggested by feedback.
            # At this point we can report/externalize that drawer2 is locked.

        else:
            print("[Exploration] Pull succeeded; drawer2 must be unlocked.")

        # To further verify, optionally attempt to open/close/operate drawer or re-check lock-known predicate
        # Or, if using exploration domain, you could invoke specialized skills to identify predicate (not required here)

        print("[Exploration] Completed exploration for missing predicate on drawer2.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()