# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided primitive skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # For demonstration, list available positions:
        print("[Info] Initial object positions:")
        for name, pos in positions.items():
            print("  {}: {}".format(name, pos))

        # ========== EXPLORATION PHASE: Identify Missing Predicate ==========
        # This phase attempts to discover which property/predicate is missing
        # before continuing the main plan. We do this by systematically
        # trying available exploratory actions and monitoring any unexpected
        # conditions or missing preconditions that block progress.

        # The relevant provided primitives are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        #  'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Strategy:
        # 1. For each critical object or location, try "execute_go" to move the robot there.
        # 2. Try picking up or interacting with objects/drawers using available skills.
        # 3. If certain actions cannot be performed (e.g., pick fails due to unknown predicate), log this.

        # Example logic: Attempt to identify missing predicates that block us.
        exploration_success = False
        missing_predicate = None

        # Try to go to and interact with each object and log failures
        for obj_name, obj_pos in positions.items():
            # Attempt to move to the object's location using 'execute_go'
            try:
                print("[Exploration] Trying to move to object '{}' at {}".format(obj_name, obj_pos))
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=None,  # Let skill_code figure out current location, if possible
                    to_location=obj_pos
                )
                print("[Exploration] Reached object '{}'.".format(obj_name))
            except Exception as e:
                print("[Exploration] Failed to move to '{}': {}".format(obj_name, str(e)))
                continue

            # Try to pick up the object
            try:
                print("[Exploration] Trying to pick up '{}'.".format(obj_name))
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj_name,
                    location=obj_pos
                )
                if done:
                    print("[Exploration] Picked up '{}' successfully.".format(obj_name))
                    continue
            except Exception as e:
                print("[Exploration] Failed to pick up '{}': {}".format(obj_name, str(e)))
                # If this step fails, note the relevant predicate or missing property.
                missing_predicate = "Failed to pick: {}".format(str(e))
                break

            # Try to interact with a drawer handle, if this is possible
            if "handle" in obj_name or "drawer" in obj_name:
                try:
                    print("[Exploration] Trying to pull '{}'.".format(obj_name))
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=obj_name,
                        handle_name=None,   # not enough info, could be inferred
                        location=obj_pos
                    )
                    if done:
                        print("[Exploration] Pulled '{}' successfully.".format(obj_name))
                except Exception as e:
                    print("[Exploration] Failed to pull '{}': {}".format(obj_name, str(e)))
                    missing_predicate = "Failed to pull: {}".format(str(e))
                    break

        if missing_predicate is not None:
            print("[Exploration] Missing Predicate or Blocker Detected: {}".format(missing_predicate))
            # If the blocker is missing known predicates (e.g., lock-known or weight-known), log and halt exploration
        else:
            print("[Exploration] No critical blockers detected in the exploration phase.")
            exploration_success = True

        # === MAIN TASK PHASE (Oracle Plan Execution placeholder) ===
        # Here, you would load and execute the oracle plan step-by-step.
        # For each action in the oracle plan, use the corresponding skill function as per the naming:
        #
        # e.g., for step in plan:
        #           if step[0] == "execute_pick":
        #               execute_pick(env, task, params...)
        #           elif step[0] == "execute_place":
        #               execute_place(env, task, params...)
        #           ...
        #
        # This placeholder assumes such plan is available and correctly formatted.

        print("[Task] Oracle plan execution phase would be handled here. (Plan integration pending.)")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
