# run_skeleton_task.py (COMPLETED for Exploration of Missing Predicate)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase to Identify Missing Predicate ---
        # Based on prompt: Attempt to execute actions involving each skill and inspect environment info for missing predicates
        # Only use the AVAILABLE skills and feedback information.

        # Get lists of known objects, drawers, locations, handles, etc. from object positions
        # For fallback, use generic keys if available
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        handles = positions.get('handles', [])
        robot_location = None

        # Try to infer current robot position (if provided)
        if 'robot' in positions and isinstance(positions['robot'], dict):
            robot_location = positions['robot'].get('location', None)
        elif 'robot' in positions:
            robot_location = positions['robot']

        # Fallback: use first location if robot position unknown
        if robot_location is None and len(locations) > 0:
            robot_location = locations[0]

        # Helper: safe call of skills, return success flag
        def try_skill_call(fn, *args, **kwargs):
            try:
                obs, reward, done = fn(env, task, *args, **kwargs)
                print(f'  [Skill: {fn.__name__}] Args: {args} | Success')
                return obs, reward, done
            except Exception as e:
                print(f'  [Skill: {fn.__name__}] Args: {args} | Error: {e}')
                return None, None, False

        # 1. Try to execute 'execute_pick' for on-floor objects
        if hasattr(env, 'on_floor_objects'):
            floor_objs = env.on_floor_objects
        else:
            # Try guessing: select objects not in any drawer
            floor_objs = []
            for o in objects:
                in_drawer = False
                if hasattr(env, 'in_drawer'):
                    for d in drawers:
                        if env.in_drawer(o, d):
                            in_drawer = True
                if not in_drawer:
                    floor_objs.append(o)
        # If no info, use all objects
        if not floor_objs and objects:
            floor_objs = objects

        # 2. For each object, try available skills to trigger predicate checks
        for obj in floor_objs:
            print(f"[Exploration] Testing object: {obj}")

            # Try picking up the object (execute_pick)
            if robot_location is not None:
                try_skill_call(execute_pick, obj, robot_location)
            else:
                print("    [Info] Unknown robot location, skipping execute_pick.")

            # Try sweeping the object (execute_sweep)
            if robot_location is not None:
                try_skill_call(execute_sweep, obj, robot_location)

        # 3. Try 'execute_place' if possible
        if drawers and floor_objs and robot_location is not None:
            d = drawers[0]
            obj = floor_objs[0]
            try:
                # Need to 'pick' first to place
                obs, reward, done = execute_pick(env, task, obj, robot_location)
                if done or not done: # Done may be False if nothing happens
                    try_skill_call(execute_place, obj, d, robot_location)
            except Exception as e:
                print(f"    [execute_place] Could not prepare: {e}")

        # 4. Try opening and closing drawer ('execute_pull' and 'execute_push')
        if drawers and handles and robot_location is not None:
            d = drawers[0]
            h = handles[0]
            try:
                # First, pick up the handle
                obs, reward, done = execute_pick(env, task, h, robot_location)
                if done or not done:
                    try_skill_call(execute_pull, d, h, robot_location)
            except Exception as e:
                print(f"    [execute_pull] Could not perform: {e}")

            try_skill_call(execute_push, d, robot_location)

        # 5. Try moving location with 'execute_go'
        if len(locations) >= 2 and robot_location is not None:
            loc_from = robot_location
            loc_to = [l for l in locations if l != loc_from]
            if loc_to:
                try_skill_call(execute_go, loc_from, loc_to[0])

        # 6. Try rotation and gripper (even if dummy)
        try:
            try_skill_call(execute_rotate)
        except Exception:
            pass
        try:
            try_skill_call(execute_gripper)
        except Exception:
            pass

        print("[Exploration] Exploration phase complete -- inspected possible actions for missing predicates.")

        # === At this point, print observations for user investigation ===
        try:
            obs_now = task.get_observation()
            print(f"  [DEBUG Observation] {obs_now}")
        except Exception as e:
            print(f"  [DEBUG] Could not retrieve observation: {e}")

        # Done with exploration. In real workflow, further logic would depend on which predicates/actions failed or succeeded.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
