# run_skeleton_task.py (Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills only as imported

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now including exploration.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dictionary mapping object names to their positions
        positions = get_object_positions()

        print("[Exploration] Begin to find missing predicates...")

        # ============================
        # === EXPLORATION PHASE ===
        # ============================
        # We explore the environment and trigger actions to look for unexpected/unknown predicates (per feedback).
        # We'll execute all skills relevant to exploration, then monitor for errors, logs, and grasp state.

        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # * Try to deduce all possible objects, drawers, and locations
        #   (Here we assume get_object_positions() keys give us object names, possibly location data)
        #   - In real code, you'd parse the observation (:init) for types.

        obj_names = []
        drawer_names = []
        location_names = set()

        for name in positions:
            if "drawer" in name:
                drawer_names.append(name)
                location_names.add(name)
            else:
                obj_names.append(name)

        if len(location_names) == 0:  # In case no drawer as location, use default
            # fallback to some default locations (assume names...)
            location_names = set(["loc_1", "loc_2", "loc_3"])

        # 1. Try to pick an object on the floor
        for obj in obj_names:
            try:
                loc = positions.get(obj, None)
                # If using simulator API, convert position to location id if needed
                loc_name = None
                for l in location_names:
                    if l in obj or obj in l:  # Simple heuristic
                        loc_name = l
                if not loc_name:
                    # Pick arbitrary location
                    loc_name = list(location_names)[0]
                print(f"[Exploration] Attempting execute_pick({obj}, {loc_name})")
                obs, reward, done = execute_pick(env, task, obj, loc_name)
            except Exception as e:
                print(f"[Exploration] execute_pick failed on {obj}: {e}")

        # 2. Try to open all drawers with all handles (execute_pull)
        for drawer in drawer_names:
            possible_handles = [o for o in obj_names if "handle" in o or "knob" in o]
            for handle in possible_handles:
                try:
                    loc_name = drawer
                    print(f"[Exploration] Attempting execute_pick({handle}, {loc_name})")
                    obs, reward, done = execute_pick(env, task, handle, loc_name)
                    print(f"[Exploration] Attempting execute_pull({drawer}, {handle}, {loc_name})")
                    obs, reward, done = execute_pull(env, task, drawer, handle, loc_name)
                except Exception as e:
                    print(f"[Exploration] execute_pull failed on {drawer},{handle}: {e}")

        # 3. Try to place held objects inside open drawers
        for drawer in drawer_names:
            try:
                loc_name = drawer
                for obj in obj_names:
                    print(f"[Exploration] Attempting execute_place({obj}, {drawer}, {loc_name})")
                    obs, reward, done = execute_place(env, task, obj, drawer, loc_name)
            except Exception as e:
                print(f"[Exploration] execute_place failed on {drawer}: {e}")

        # 4. Try to push all drawers (to close them)
        for drawer in drawer_names:
            try:
                loc_name = drawer
                print(f"[Exploration] Attempting execute_push({drawer}, {loc_name})")
                obs, reward, done = execute_push(env, task, drawer, loc_name)
            except Exception as e:
                print(f"[Exploration] execute_push failed on {drawer}: {e}")

        # 5. Try to go to every pair of locations
        loc_list = list(location_names)
        for i in range(len(loc_list)):
            for j in range(len(loc_list)):
                if i == j:
                    continue
                try:
                    print(f"[Exploration] Attempting execute_go({loc_list[i]}, {loc_list[j]})")
                    obs, reward, done = execute_go(env, task, loc_list[i], loc_list[j])
                except Exception as e:
                    print(f"[Exploration] execute_go failed between {loc_list[i]}, {loc_list[j]}: {e}")

        # 6. Try execute_sweep and execute_gripper on all objects & locations (to check effect)
        for obj in obj_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting execute_sweep({obj}, {loc})")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed on {obj} at {loc}: {e}")
        try:
            print(f"[Exploration] Attempting execute_gripper()")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Finished exploration phase for missing predicates.")
        # After running, check logs/errors for hints about missing predicates

        # =========================
        # === MAIN PLAN PHASE ===
        # =========================

        print("[Main Task] (Placeholder): Insert your plan execution here using available skills.")
        # --- EXAMPLE PLAN EXECUTION FLOW ---
        # 1. Move to required location (use execute_go)
        # 2. Pick object (execute_pick)
        # 3. Pull drawer, push drawer, place, etc as required
        # Replace below example with your actual plan steps if you have the plan

        # Example:
        # try:
        #     obs, reward, done = execute_go(env, task, "start_location", "goal_location")
        #     obs, reward, done = execute_pick(env, task, "object_a", "goal_location")
        #     obs, reward, done = execute_pull(env, task, "drawer_a", "handle_a", "goal_location")
        #     obs, reward, done = execute_place(env, task, "object_a", "drawer_a", "goal_location")
        # except Exception as e:
        #     print(f"[Main Task] Plan execution error: {e}")

        # TODO: Replace with actual, dynamically generated or oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
