# run_skeleton_task.py (Completed Skeleton with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions or environment info
        positions = get_object_positions()

        # -------------------------------------------
        # === EXPLORATION PHASE FOR MISSING PREDICATE ===
        # -------------------------------------------
        # The exploration domain and feedback indicate that we may need to discover
        # which attribute (predicate) is missing (e.g., lock-known, temperature-known, etc).
        #
        # Therefore, we systematically use the available skills to trigger effects
        # and examine the observations to deduce the missing one.
        #
        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 
        #                   'execute_pull', 'execute_sweep', 'execute_rotate', 
        #                   'execute_go', 'execute_gripper']
        #
        # For exploration, primarily use:
        # - execute_go (move/relocate)
        # - execute_pick (pick up object)
        # - execute_pull (pull, related in exploration domain to lock-known)
        #
        # If action fails unexpectedly, it may indicate required predicates.

        # Find all locations, drawers, objects from observation (positions and descriptions)
        location_names = []
        drawer_names = []
        object_names = []
        handle_objects = []  # objects that are handles for drawers
        # Attempt to extract from positions, fallback to descriptions in obs if needed
        for name in positions:
            if "location" in name or "loc" in name:
                location_names.append(name)
            if "drawer" in name:
                drawer_names.append(name)
            if "handle" in name:
                handle_objects.append(name)
            if name.startswith("obj") or "object" in name:
                object_names.append(name)
        
        # Fallback: use type information if location/object arrays are empty
        if not location_names or not object_names:
            # Try to extract from obs if provided as a dict or descriptions
            if hasattr(obs, 'keys'):
                for k in obs.keys():
                    if "location" in k:
                        location_names.append(k)
                    if "drawer" in k:
                        drawer_names.append(k)
                    if "handle" in k:
                        handle_objects.append(k)
                    if "object" in k or k.startswith("obj"):
                        object_names.append(k)
        
        # If empty, log warning but continue
        if not location_names:
            print(" [Warn] Could not infer location names from positions/obs.")
        if not object_names:
            print(" [Warn] Could not infer object names from positions/obs.")
        if not drawer_names:
            print(" [Warn] Could not infer drawer names from positions/obs.")

        # 1. Try moving the robot to all locations using 'execute_go'
        robot_location = None
        # Deduce initial location
        for name in location_names:
            # If the robot is already at a certain location, set it
            if "robot" in positions and positions["robot"] == positions[name]:
                robot_location = name
                break
        if not robot_location and location_names:
            robot_location = location_names[0]  # arbitrary
        
        explored_predicates = set()
        exploration_logs = []

        print("[Exploration] Testing 'execute_go', 'execute_pick', 'execute_pull' to identify required predicates.")

        # 2. Try picking each object at its position (simulate hand-empty, robot-free, robot-at <place>)
        for obj in object_names:
            # Assume all actions take place at the first location as default
            for loc in location_names[:1]:  # Try just the first location for each for exploration
                try:
                    print(f"[Exploration] Attempting to pick object {obj} at location {loc}...")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj,
                        loc
                    )
                    print(f"[Exploration] execute_pick({obj}, {loc}) succeeded.")
                    exploration_logs.append(f"Picked {obj} at {loc}.")
                    explored_predicates.add("holding")
                except Exception as e:
                    print(f"[Exploration] execute_pick({obj}, {loc}) failed: {e}")
                    exploration_logs.append(f"Pick {obj} at {loc} FAILED: {e}")

        # 3. Try pulling handles to see if lock-known is an issue (exploration domain: pull reveals lock-known)
        for d in drawer_names:
            # Find a handle object for this drawer
            for h in handle_objects:
                try:
                    for loc in location_names[:1]:
                        print(f"[Exploration] Attempting to pull handle {h} of drawer {d} at location {loc}...")
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            d,
                            h,
                            loc
                        )
                        print(f"[Exploration] execute_pull({d}, {h}, {loc}) succeeded.")
                        exploration_logs.append(f"Pulled handle {h} of {d} at {loc}.")
                        explored_predicates.add("drawer-open")
                except Exception as e:
                    print(f"[Exploration] execute_pull({d}, {h}, {loc}) failed: {e}")
                    exploration_logs.append(f"Pull handle {h} of {d} at {loc} FAILED: {e}")

        # 4. Try placing each picked object into the first drawer at the first location (test place preconditions)
        for o in object_names:
            for d in drawer_names[:1]:
                for loc in location_names[:1]:
                    try:
                        print(f"[Exploration] Attempting to place object {o} into drawer {d} at {loc}...")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            o,
                            d,
                            loc
                        )
                        print(f"[Exploration] execute_place({o}, {d}, {loc}) succeeded.")
                        exploration_logs.append(f"Placed {o} in {d} at {loc}.")
                        explored_predicates.add("in-drawer")
                    except Exception as e:
                        print(f"[Exploration] execute_place({o}, {d}, {loc}) failed: {e}")
                        exploration_logs.append(f"Place {o} in {d} at {loc} FAILED: {e}")

        print("[Exploration] Exploration done. Explored predicates:", explored_predicates)
        print("[Exploration] Action logs:")
        for log in exploration_logs:
            print("   ", log)
        
        # At this point, based on success/failure of actions, you can check logs to see, 
        # for example, if an action failed with a message indicating a missing predicate like 
        # "lock-known" (in case of drawer), "identified", etc.

        # -------------------------------------------
        # === END EXPLORATION PHASE ===
        # (In production/planning, you would now use this knowledge for the actual plan.)
        # -------------------------------------------

        # You can now proceed to execute an oracle plan by referencing the skills directly,
        # provided you know the required objects, drawers, locations, and handle mapping.
        #
        # For this generic template, the core of the task is the exploration above.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
