# run_skeleton_task.py (Executable Code including Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def explore_environment(env, task, positions, available_skills):
    """
    Exploration phase: Use only available skills to try and determine if an action is blocked due to missing predicates, 
    as indicated by the feedback. The purpose is to systematically test skills and uncover which predicates are required.
    """
    # Create a simple log
    exploration_log = []
    print(">>> [Exploration] Starting predicate exploration...")

    # Example: Try each skill in order, catch any errors that may indicate a missing predicate
    for skill_name in available_skills:
        try:
            # Each skill has different parameters; use dummy/first object/positions to test
            if skill_name == 'execute_pick':
                # Find a free object on the floor; get any first available
                obj_on_floor = next((o for o, v in positions.items() if v.get('on_floor')), None)
                p_loc = next((v['location'] for o, v in positions.items() if v.get('on_floor')), None)
                if obj_on_floor and p_loc:
                    print(f"[Exploration] Trying execute_pick for object {obj_on_floor} at {p_loc}")
                    obs, reward, done = execute_pick(
                        env, task,
                        obj_on_floor, p_loc
                    )
                    exploration_log.append(('execute_pick', True))
            elif skill_name == 'execute_place':
                # Find an object that is currently held (simplified guess)
                held_obj = next((o for o, v in positions.items() if v.get('holding')), None)
                drawer = next((o for o, v in positions.items() if v.get('drawer_open')), None)
                robot_loc = next((v['location'] for o, v in positions.items() if 'location' in v), None)
                if held_obj and drawer and robot_loc:
                    print(f"[Exploration] Trying execute_place for object {held_obj} in drawer {drawer} at {robot_loc}")
                    obs, reward, done = execute_place(
                        env, task, held_obj, drawer, robot_loc
                    )
                    exploration_log.append(('execute_place', True))
            elif skill_name == 'execute_push':
                # Find any open drawer and its location
                open_drawer = next((o for o, v in positions.items() if v.get('drawer_open')), None)
                robot_loc = next((v['location'] for o, v in positions.items() if 'location' in v), None)
                if open_drawer and robot_loc:
                    print(f"[Exploration] Trying execute_push for drawer {open_drawer} at {robot_loc}")
                    obs, reward, done = execute_push(
                        env, task, open_drawer, robot_loc
                    )
                    exploration_log.append(('execute_push', True))
            elif skill_name == 'execute_pull':
                # Find a handle object, its associated drawer, and test pulling
                h_obj = next((o for o, v in positions.items() if v.get('is_handle')), None)
                drawer = next((o for o, v in positions.items() if v.get('drawer_closed') and v.get('drawer_unlocked')), None)
                robot_loc = next((v['location'] for o, v in positions.items() if 'location' in v), None)
                if h_obj and drawer and robot_loc:
                    print(f"[Exploration] Trying execute_pull for drawer {drawer} handle {h_obj} at {robot_loc}")
                    obs, reward, done = execute_pull(
                        env, task, drawer, h_obj, robot_loc
                    )
                    exploration_log.append(('execute_pull', True))
            elif skill_name == 'execute_go':
                # Find two locations and move
                locs = [v['location'] for o, v in positions.items() if 'location' in v]
                if len(locs) > 1:
                    print(f"[Exploration] Trying execute_go from {locs[0]} to {locs[1]}")
                    obs, reward, done = execute_go(
                        env, task, locs[0], locs[1]
                    )
                    exploration_log.append(('execute_go', True))
            elif skill_name == 'execute_sweep':
                # Try to sweep any object on the floor
                obj_on_floor = next((o for o, v in positions.items() if v.get('on_floor')), None)
                p_loc = next((v['location'] for o, v in positions.items() if v.get('on_floor')), None)
                if obj_on_floor and p_loc:
                    print(f"[Exploration] Trying execute_sweep for object {obj_on_floor} at {p_loc}")
                    obs, reward, done = execute_sweep(
                        env, task, obj_on_floor, p_loc
                    )
                    exploration_log.append(('execute_sweep', True))
            elif skill_name == 'execute_gripper':
                print(f"[Exploration] Trying execute_gripper")
                obs, reward, done = execute_gripper(
                    env, task
                )
                exploration_log.append(('execute_gripper', True))
            # You can add more detailed logging or more test alternatives if the state is ambiguous
        except Exception as e:
            print(f"[Exploration][{skill_name}] Exception: {type(e).__name__}: {e}")
            exploration_log.append((skill_name, False, str(e)))
            continue

    print(">>> [Exploration] Exploration finished. See log for results.")
    for entry in exploration_log:
        print("  [LOG]", entry)

    return exploration_log

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Returns a dictionary: key = object id/name, value = dict describing state/position
        positions = get_object_positions()

        # === EXPLORATION PHASE FOR MISSING PREDICATE ===
        # Use only available skills to try to expose missing predicates
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]
        explore_environment(env, task, positions, available_skills)

        # === MAIN TASK/PLAN EXECUTION ===
        # After exploration (and possible manual intervention), proceed with the actual oracle plan.
        # This phase is a placeholder: you should load or hardcode your real plan here.
        # Sample pseudo-plan (you would replace this with your actual plan logic):

        try:
            # 1. Move to location of object
            obj_to_pick = next((o for o, v in positions.items() if v.get('on_floor')), None)
            pick_location = positions[obj_to_pick]['location'] if obj_to_pick else None
            if obj_to_pick and pick_location:
                print(f"[Task] Moving to location {pick_location} for {obj_to_pick}")
                obs, reward, done = execute_go(env, task, None, pick_location)  # None as 'from', if known, fill in
                # 2. Pick the object
                print(f"[Task] Picking {obj_to_pick} at {pick_location}")
                obs, reward, done = execute_pick(env, task, obj_to_pick, pick_location)

            # 3. Move to drawer's location
            drawer = next((o for o, v in positions.items() if v.get('drawer_open') or v.get('drawer_closed')), None)
            drawer_location = positions[drawer]['location'] if drawer else None
            if drawer and drawer_location:
                print(f"[Task] Moving to drawer {drawer} at {drawer_location}")
                obs, reward, done = execute_go(env, task, pick_location, drawer_location)

                # 4. If drawer closed, try to pull open
                if positions[drawer].get('drawer_closed'):
                    handle_obj = next((o for o, v in positions.items() if v.get('is_handle') and v.get('handle_of') == drawer), None)
                    print(f"[Task] Pulling open {drawer} using handle {handle_obj}")
                    obs, reward, done = execute_pull(env, task, drawer, handle_obj, drawer_location)

                # 5. Place object in drawer
                print(f"[Task] Placing {obj_to_pick} in drawer {drawer}")
                obs, reward, done = execute_place(env, task, obj_to_pick, drawer, drawer_location)

                # 6. Push drawer closed
                print(f"[Task] Pushing (closing) drawer {drawer}")
                obs, reward, done = execute_push(env, task, drawer, drawer_location)

            print("[Task] Oracle plan executed (or sample plan executed).")
        except Exception as e:
            print(f"[Task][Execution] Exception encountered: {type(e).__name__}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
