# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # The keys and structure of positions depend on environment config and observation.

        # --- EXPLORATION PHASE: Attempt to discover/confirm required predicates ---
        #
        # Feedback suggests a missing predicate in the planning domain. To resolve this,
        # we run an exploration phase using the available skills to try to surface unknown or missing predicates
        # by probing environment transitions, especially with predicates like lock-known, identified, etc., as in the provided exploration domain knowledge.
        #
        # The idea is to probe (e.g. by attempting pulls, picks, etc.) and examine the transitions/state to see what is learnable,
        # thereby surfacing "missing" predicates utilized in planning.

        print("[Exploration] Starting exploration phase to identify required predicates.")

        # --- Example logic: Iterate through accessible objects/locations to probe for hidden properties ---
        robot_loc = None
        # Guess the robot's initial location based on positions or description if possible
        for key in positions:
            if "robot" in key or "base" in key:
                robot_loc = positions[key]
                break

        if robot_loc is None:
            # Fallback, try to fetch from description if key not found
            if hasattr(task, 'get_robot_location'):
                robot_loc = task.get_robot_location()
            else:
                print("[Warning] Could not determine robot initial location.")

        # For illustration: suppose we need to iterate over all discovered objects/handles
        objects = []
        handles = []
        drawers = []
        for k, v in positions.items():
            if 'handle' in k.lower():
                handles.append(k)
            elif 'drawer' in k.lower():
                drawers.append(k)
            elif 'object' in k.lower() or 'obj' in k.lower() or k.startswith('ball') or k.startswith('cube'):
                objects.append(k)

        # Use all available skills for active exploration
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For each drawer, try opening with pulls to possibly discover lock-known, if exploration actions mirror skill names
        for dname in drawers:
            drawer_pos = positions[dname]
            # Try each handle supposedly attached to drawer
            for hname in handles:
                # Try moving to the drawer/handle location
                try:
                    # Use execute_go skill to move to drawer/handle position, if available
                    obs, reward, done = execute_go(env, task, from_position=robot_loc, to_position=drawer_pos)
                    robot_loc = drawer_pos  # update
                except Exception as e:
                    print(f"[Exploration] execute_go failed for {dname} ({e})")
                    continue
                # Try to pick up the handle (simulate grasp/hold)
                try:
                    obs, reward, done = execute_pick(env, task, obj_name=hname, location_name=dname)
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for handle {hname} on drawer {dname} ({e})")
                    continue
                # Try pulling the drawer (may yield lock-known or related predicate)
                try:
                    obs, reward, done = execute_pull(env, task, drawer_name=dname, handle_name=hname, location_name=dname)
                    if done:
                        print(f"[Exploration] Pulled {dname} with {hname} - checking for lock-known or other predicate discovery.")
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {dname} ({e})")

        # Try to pick and place generic objects to see if weight-known, durability-known, etc., are discovered
        for oname in objects:
            obj_pos = positions[oname]
            try:
                obs, reward, done = execute_pick(env, task, obj_name=oname, location_name=obj_pos)
            except Exception as e:
                print(f"[Exploration] execute_pick failed for obj {oname} ({e})")
                continue
            # Try to place in any available drawer if exist
            for dname in drawers:
                drawer_pos = positions[dname]
                try:
                    obs, reward, done = execute_place(env, task, obj_name=oname, drawer_name=dname, location_name=drawer_pos)
                except Exception as e:
                    print(f"[Exploration] execute_place failed: {oname} to {dname} ({e})")
                    continue

        # Exploration for sweep or gripper if relevant (some actions are stubs; call anyway)
        try:
            for oname in objects:
                obs, reward, done = execute_sweep(env, task, obj_name=oname, location_name=positions[oname])
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        try:
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # Initial exploration done.
        print("[Exploration] Exploration phase finished. Proceed to planned task execution if predicates are resolved.")

        # TODO: After exploration, proceed with actual planned sequence (oracle plan).
        # In a practical implementation, an oracle plan would be parsed and the following loop would execute step-by-step actions.

        # === Example Task Plan (Replace with actual oracle plan logic) ===
        # (Below is an illustrative pattern; actual action call parameters must match those of imported skills)

        print("[Task] No oracle plan was provided, so only exploration performed for predicate discovery.")

        # If an oracle plan were provided, it might look like:
        # for step in oracle_plan:
        #     try:
        #         if step['action'] == 'execute_pick':
        #             execute_pick(env, task, obj_name=step['object'], location_name=step['location'])
        #         elif step['action'] == 'execute_place':
        #             execute_place(env, task, obj_name=step['object'], drawer_name=step['drawer'], location_name=step['location'])
        #         elif step['action'] == 'execute_push':
        #             execute_push(env, task, drawer_name=step['drawer'], location_name=step['location'])
        #         elif step['action'] == 'execute_pull':
        #             execute_pull(env, task, drawer_name=step['drawer'], handle_name=step['handle'], location_name=step['location'])
        #         elif step['action'] == 'execute_go':
        #             execute_go(env, task, from_position=step['from'], to_position=step['to'])
        #         elif step['action'] == 'execute_sweep':
        #             execute_sweep(env, task, obj_name=step['object'], location_name=step['location'])
        #         elif step['action'] == 'execute_gripper':
        #             execute_gripper(env, task)
        #     except Exception as e:
        #         print(f"[Task Execution] Exception during action {step['action']}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()