# run_skeleton_task.py (Executable Code with Exploration Phase and Plan Execution)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use skill functions directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with missing predicate discovery.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (Optional) Initialize video logging for your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()  # Dictionary: object names as keys, positions as values

        # === PHASE 1: EXPLORATION - Discover Missing Predicates ===
        # Feedback: (drawer-unlocked drawer_middle) was missing. We must perform exploration to check unlocked state.

        print("[Exploration] Starting exploration phase to determine missing predicates (e.g., drawer lock status).")

        # Determine names (You might need to adapt if object naming is not canonical)
        drawer_candidates = [name for name in positions.keys() if 'drawer' in name]
        handle_candidates = [name for name in positions.keys() if 'handle' in name]
        location_candidates = [name for name in positions.keys() if ('table' in name) or ('location' in name)]

        robot_at_loc = None
        if location_candidates:
            robot_at_loc = location_candidates[0]  # Assume robot starts here

        discovered_drawer_states = {}

        for drawer in drawer_candidates:
            # Find matching handle for drawer, if exists
            matching_handle = None
            for h in handle_candidates:
                if drawer in h:  # e.g., 'drawer_middle' in 'handle_drawer_middle'
                    matching_handle = h
                    break
            if not matching_handle and handle_candidates:
                matching_handle = handle_candidates[0]  # fallback
            
            drawer_pos = positions[drawer]
            handle_pos = None
            if matching_handle:
                handle_pos = positions[matching_handle]
            else:
                continue  # Cannot proceed without handle

            # Move to handle location if needed
            current_loc = robot_at_loc
            if current_loc and positions.get(current_loc) != handle_pos:
                try:
                    obs, reward, done = execute_go(env, task, current_loc, matching_handle)
                    robot_at_loc = matching_handle
                except Exception as e:
                    print(f"[Exploration] Warning: execute_go failed: {e}")

            # Try to pick up the handle
            try:
                obs, reward, done = execute_pick(env, task, matching_handle, robot_at_loc)
                if done:
                    print("[Exploration] Task ended during pick (unexpected).")
                    return
            except Exception as e:
                print(f"[Exploration] Warning: execute_pick failed: {e}")

            # Try to pull (to determine lock state) - this is the exploration primitive
            try:
                obs, reward, done = execute_pull(env, task, drawer, matching_handle, robot_at_loc)
                # If successful, this drawer is unlocked and open now
                discovered_drawer_states[drawer] = 'unlocked'
                print(f"[Exploration] Drawer {drawer} can be pulled open (UNLOCKED).")
                # For test, push it back to closed (if required by task)
                try:
                    obs, reward, done = execute_push(env, task, drawer, robot_at_loc)
                except Exception:
                    pass
            except Exception as e:
                # Could not pull - likely locked
                discovered_drawer_states[drawer] = 'locked'
                print(f"[Exploration] Drawer {drawer} cannot be pulled open (LOCKED): {e}")

            # Place back handle, just in case
            # Typically, place is for objects to a drawer, skip if not needed

        print(f"[Exploration] Drawer states after exploration: {discovered_drawer_states}")

        # === PHASE 2: PLAN EXECUTION ===

        # Example oracle plan, you must adapt this to your environment and task
        # Set up target objects/locations as appropriate

        print("[Task] --- Beginning Oracle Plan Execution ---")

        # Example plan: Open the unlocked drawer and place an object inside

        # Assume you want to place 'red_cube' into 'drawer_middle' if it's unlocked

        target_drawer = 'drawer_middle' if 'drawer_middle' in discovered_drawer_states else None
        if target_drawer is None:
            print("[Task] No drawer_middle found in exploration. Task aborted.")
            return

        if discovered_drawer_states[target_drawer] != 'unlocked':
            print(f"[Task] {target_drawer} is locked! Cannot proceed with oracle plan.")
            return

        # Find a cube to pick and place (e.g., red_cube)
        cube_candidates = [n for n in positions if 'cube' in n]
        if not cube_candidates:
            print("[Task] No cube object found to manipulate. Task aborted.")
            return
        target_cube = cube_candidates[0]
        cube_pos = positions[target_cube]

        # Find handle for the drawer
        drawer_handle = None
        for h in handle_candidates:
            if target_drawer in h:
                drawer_handle = h
                break
        if not drawer_handle:
            print("[Task] No handle found for drawer. Task aborted.")
            return

        handle_pos = positions[drawer_handle]

        # Move to cube position, pick the cube
        try:
            if robot_at_loc != target_cube:
                obs, reward, done = execute_go(env, task, robot_at_loc, target_cube)
                robot_at_loc = target_cube
            obs, reward, done = execute_pick(env, task, target_cube, robot_at_loc)
            if done:
                print("[Task] Task ended at pick cube.")
                return
        except Exception as e:
            print(f"[Task] Error during picking up cube: {e}")
            return

        # Move to handle and open the drawer (assume closed)
        try:
            if robot_at_loc != drawer_handle:
                obs, reward, done = execute_go(env, task, robot_at_loc, drawer_handle)
                robot_at_loc = drawer_handle
            obs, reward, done = execute_pick(env, task, drawer_handle, robot_at_loc)
            if done:
                print("[Task] Task ended at pick handle.")
                return
            obs, reward, done = execute_pull(env, task, target_drawer, drawer_handle, robot_at_loc)
            if done:
                print("[Task] Task ended at open drawer.")
                return
        except Exception as e:
            print(f"[Task] Error during opening drawer: {e}")
            return

        # Move to position for place, place cube into drawer
        try:
            # Put at drawer location for place (simulate that the drawer and robot are at the same spot)
            obs, reward, done = execute_place(env, task, target_cube, target_drawer, robot_at_loc)
            if done:
                print("[Task] Task ended after placing cube in drawer.")
                return
        except Exception as e:
            print(f"[Task] Error during placing cube: {e}")

        print("[Task] --- Oracle Plan Execution Complete ---")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()