# run_skeleton_task.py (Complete, Exploration Phase Included)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All skills provided externally; do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ====== Exploration Phase to Find Missing Predicate ======
        # The goal is to systematically explore the environment and interact with objects/drawers to determine 
        # if there are unknown states or predicates needed for the plan (i.e., whether we are missing 
        # knowledge about lock, weight, temperature, durability, etc. for objects).
        
        # We'll use the available skills ('execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper')
        # and the provided domain knowledge to actively probe for missing predicates like lock status etc.

        # 1. List of drawers and objects (if available from environment description)
        try:
            # Extract all objects & drawers if possible
            object_names = []
            drawer_names = []
            handle_names = []
            location_names = []
            # The descriptions variable, if structured, may provide this. We'll check in positions as a fallback.
            for name, pos in positions.items():
                if 'drawer' in name:
                    drawer_names.append(name)
                elif 'handle' in name:
                    handle_names.append(name)
                elif 'location' in name or 'pos' in name or 'place' in name or name.startswith('loc'):
                    location_names.append(name)
                else:
                    object_names.append(name)
        except Exception as e:
            print("[Exploration] Couldn't distinguish objects/drawers:", e)
            object_names = list(positions.keys())
            drawer_names = []
            handle_names = []
            location_names = []

        # If no locations, try to infer at least one location from the robot position or any object
        if not location_names:
            # Try to fetch robot location from obs or from 'robot' or 'base' in positions
            for k in positions:
                if 'robot' in k or 'base' in k or 'loc' in k or 'place' in k:
                    location_names.append(k)
            # Fallback: just pick any
            if not location_names and positions:
                location_names.append(list(positions.keys())[0])

        # For exploration, we simulate going to each location and performing pull/open actions if available,
        # checking for exceptions or unknowns that would indicate missing predicates

        # Pick one start location (assume robot starts here)
        robot_current_location = None
        for k in location_names:
            if 'robot' in k or 'base' in k:
                robot_current_location = k
                break
        if robot_current_location is None:
            robot_current_location = location_names[0]

        print("[Exploration] Locations:", location_names)
        print("[Exploration] Objects:", object_names)
        print("[Exploration] Drawers:", drawer_names)
        print("[Exploration] Handles:", handle_names)
        print("[Exploration] Robot location:", robot_current_location)

        missing_predicate_found = None

        # Start the exploration loop
        for location in location_names:
            if location != robot_current_location:
                try:
                    print(f"[Exploration] execute_go: {robot_current_location} -> {location}")
                    # Assuming execute_go(env, task, from_location, to_location)
                    obs, reward, done = execute_go(env, task, robot_current_location, location)
                    robot_current_location = location
                except Exception as e:
                    print(f"[Exploration] Could not move to {location}: {e}")

            # Try to pick up and pull any object/handle in this location to probe for missing predicates
            for obj in object_names + handle_names:
                try:
                    # First try picking the object if on-floor and hand is empty
                    print(f"[Exploration] Trying execute_pick on object {obj} at {location}")
                    obs, reward, done = execute_pick(env, task, obj, location)
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {location}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {location}: {str(e)}")
                    # If error indicates missing precondition/predicate, note that
                    msg = str(e).lower()
                    if 'locked' in msg or 'unlock' in msg or 'lock-known' in msg:
                        missing_predicate_found = 'lock-known'
                    elif 'weight-known' in msg:
                        missing_predicate_found = 'weight-known'
                    elif 'durability-known' in msg:
                        missing_predicate_found = 'durability-known'
                    elif 'temperature-known' in msg:
                        missing_predicate_found = 'temperature-known'
                    elif 'identified' in msg:
                        missing_predicate_found = 'identified'

                # Try to pull the object (for handles/drawers)
                try:
                    print(f"[Exploration] Trying execute_pull on object {obj} at {location}")
                    obs, reward, done = execute_pull(env, task, obj, obj, location)
                    if done:
                        print(f"[Exploration] Successfully pulled {obj} at {location}")
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {obj} at {location}: {str(e)}")
                    # Similarly judge missing predicates
                    msg = str(e).lower()
                    if 'locked' in msg or 'unlock' in msg or 'lock-known' in msg:
                        missing_predicate_found = 'lock-known'
                    elif 'weight-known' in msg:
                        missing_predicate_found = 'weight-known'
                    elif 'durability-known' in msg:
                        missing_predicate_found = 'durability-known'
                    elif 'temperature-known' in msg:
                        missing_predicate_found = 'temperature-known'
                    elif 'identified' in msg:
                        missing_predicate_found = 'identified'

                # As a fallback, try any other skill that may probe predicates
                try:
                    print(f"[Exploration] Trying execute_sweep on object {obj} at {location}")
                    obs, reward, done = execute_sweep(env, task, obj, location)
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {location}: {str(e)}")

        # Report result      
        if missing_predicate_found:
            print(f"[Exploration] Missing predicate identified in environment: {missing_predicate_found}")
        else:
            print("[Exploration] No missing predicates detected in current exploration.")

        # You can now update your PDDL/problem, or plan accordingly

        # === (After Exploration, Proceed with Oracle Task Plan) ===
        # --- Insert oracle (provided) task plan execution here ---
        #
        # Example (pseudo-steps, replace/implement with actual logic for your domain and oracle plan):
        #
        # for action in oracle_plan:
        #     try:
        #         if action["type"] == "execute_go":
        #             obs, reward, done = execute_go(env, task, action['from'], action['to'])
        #         elif action["type"] == "execute_pick":
        #             obs, reward, done = execute_pick(env, task, action["obj"], action["loc"])
        #         elif action["type"] == "execute_place":
        #             obs, reward, done = execute_place(env, task, action["obj"], action["drawer"], action["loc"])
        #         # ... etc. for each action/skill
        #         if done:
        #             print(f"[Task] Done after {action['type']}")
        #             break
        #     except Exception as e:
        #         print(f"[Task] Error during {action['type']}: {e}")
        #
        # (The detailed oracle plan steps are to be inserted after integrating exploration output.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()