# run_skeleton_task.py (Completed for Predicate Exploration Task)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions = {'drawer': (x1,y1,z1), 'drawer_top_handle': (x2,y2,z2), ...}
        positions = get_object_positions()

        # Begin Exploration phase to determine missing predicate, based on feedback
        # Feedback says: (on-floor drawer_top_handle)
        # This suggests the missing predicate relates to the placement / property of drawer_top_handle
        # We need to perform an exploration to clarify the nature of "on-floor drawer_top_handle"

        # STEP 1: Move robot to the location of drawer_top_handle (if possible)
        # This is to ensure we can inspect/operate on the handle
        # We assume available skill: execute_go (move command)
        
        handle_name = None
        handle_pos = None
        # Try to identify a handle object in positions
        for name in positions:
            if 'handle' in name:
                handle_name = name
                handle_pos = positions[name]
                break

        if handle_name is None or handle_pos is None:
            print("[Exploration] Could not find a drawer handle in object positions.")
        else:
            print(f"[Exploration] Found handle '{handle_name}' at position {handle_pos}.")

            # Find robot's current location: assuming 'robot' key or 'robot-at' property is in positions
            robot_loc = None
            if 'robot' in positions:
                robot_loc = positions['robot']
            elif 'robot-at' in positions:
                robot_loc = positions['robot-at']
            else:
                # Fallback: Not specified, use None
                robot_loc = None

            # These variables are for demonstration; actual movement may require named locations.
            # Try to get named locations from the environment
            from_loc = None
            to_loc = None
            loc_names = [k for k in positions.keys() if 'loc' in k or 'room' in k or 'table' in k]
            if loc_names:
                # Just pick two
                from_loc = loc_names[0]
                to_loc = loc_names[1] if len(loc_names)>1 else loc_names[0]
            else:
                from_loc = "start"
                to_loc = "handle_area"

            # 1. Move the robot near the handle (execute_go skill)
            try:
                obs1, reward1, done1 = execute_go(
                    env, 
                    task,
                    from_loc,
                    to_loc, 
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

            # 2. Try to pick up the drawer_top_handle (execute_pick skill)
            try:
                obs2, reward2, done2 = execute_pick(
                    env,
                    task,
                    handle_name,
                    to_loc,
                    timeout=10.0
                )
                print(f"[Exploration] Attempted picking '{handle_name}' at '{to_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

            # 3. Try to pull (execute_pull skill) on the handle
            # For explore/diagnose missing lock-known/on-floor facts
            try:
                # The handle may have an associated drawer; try to extract if available
                drawer_name = None
                for obj in positions:
                    if 'drawer' in obj and obj != handle_name:
                        drawer_name = obj
                        break
                if drawer_name is None:
                    drawer_name = "drawer"  # fallback

                obs3, reward3, done3 = execute_pull(
                    env,
                    task,
                    drawer_name,
                    handle_name,
                    to_loc,
                    timeout=10.0
                )
                print(f"[Exploration] Executed pull on drawer '{drawer_name}' with handle '{handle_name}' at '{to_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

            # 4. Place the handle back or on/in the drawer if necessary (execute_place)
            try:
                obs4, reward4, done4 = execute_place(
                    env,
                    task,
                    handle_name,
                    drawer_name,
                    to_loc,
                    timeout=10.0
                )
                print(f"[Exploration] Placed '{handle_name}' into '{drawer_name}' at '{to_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

            # 5. Push the drawer closed (if applicable, execute_push)
            try:
                obs5, reward5, done5 = execute_push(
                    env,
                    task,
                    drawer_name,
                    to_loc,
                    timeout=10.0
                )
                print(f"[Exploration] Pushed drawer '{drawer_name}' closed at '{to_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

            # 6. Try sweeping the handle/object to see environment response (execute_sweep)
            try:
                obs6, reward6, done6 = execute_sweep(
                    env,
                    task,
                    handle_name,
                    to_loc,
                    timeout=10.0
                )
                print(f"[Exploration] Swept '{handle_name}' at '{to_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

            # 7. Try running gripper primitive (execute_gripper)
            try:
                obs7, reward7, done7 = execute_gripper(
                    env,
                    task,
                    timeout=5.0
                )
                print(f"[Exploration] Ran gripper skill.")
            except Exception as e:
                print(f"[Exploration] execute_gripper failed: {e}")

        # Main message about missing predicate exploration:
        print("[Exploration] Exploration phase completed.")
        print("[Exploration] Based on feedback (on-floor drawer_top_handle),")
        print("             check whether the handle is treated as an object on the floor or needs a special predicate (e.g., attached-to, on-floor, etc.).")
        print("             If robot cannot pick/manipulate the handle as expected,")
        print("             the missing or incorrect predicate should be reported and incorporated in the domain/problem description.")


    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
