# run_skeleton_task.py (Completed Executable Exploration & Plan Executor)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only these primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Objects & Locations Layout (based on observation/init) ===
        # You must adapt these according to your actual observation!
        # For illustration, we'll use typical keys that are likely to exist in RLBench setup.

        # Try to extract handles from the positions dictionary
        object_names = list(positions.keys())
        drawer_names = [name for name in object_names if 'drawer' in name]
        handle_names = [name for name in object_names if 'handle' in name]
        object_on_floor_names = [name for name in object_names if 'object' in name or 'item' in name]
        location_names = [name for name in object_names if 'location' in name or 'pos' in name or 'goal' in name]

        # Fallbacks/Assumptions for position keys
        robot_initial_pos = None
        if 'robot' in positions:
            robot_initial_pos = positions['robot']
        elif len(location_names) > 0:
            robot_initial_pos = positions[location_names[0]]

        # === --- Start of Exploration Phase --- ===
        # Based on feedback: must explore to discover which predicate is missing!
        #
        # We'll sequentially try executing all available skills with plausible arguments, 
        # watching for errors or effects. Particularly, 'execute_pull' may reveal a missing predicate.

        print("[Exploration] Starting exploration phase to identify missing predicate or unlock state.")

        # For each drawer, handle, try executing pull, push, pick, place, etc.
        for drawer in drawer_names:
            # --- Try opening drawer. Requires handle. ---
            # Find matching handle
            handle_match = None
            for handle in handle_names:
                # Typical RLBench convention: handle-of via underscores, e.g., "drawer1_handle"
                if drawer in handle or handle in drawer:
                    handle_match = handle
                    break
            if handle_match is None and handle_names:
                handle_match = handle_names[0]  # fallback: just pick one

            # --- Find a plausible location nearby ---
            # For RLBench, drawer position, or location_? positions, or just use handle's position
            if handle_match and handle_match in positions:
                robot_pos = positions[handle_match]
            elif drawer in positions:
                robot_pos = positions[drawer]
            elif robot_initial_pos is not None:
                robot_pos = robot_initial_pos
            else:
                robot_pos = None

            # Begin attempting skill execution:
            try:
                # 1. Move robot to drawer location (execute_go), if available
                if robot_pos is not None and 'execute_go' in globals():
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_pos=robot_initial_pos,
                        to_pos=robot_pos
                    )
                    print(f"[Exploration] execute_go to {drawer} ({robot_pos}) succeeded.")
                # 2. Pick the handle (execute_pick)
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target=handle_match,
                    location=robot_pos
                )
                print(f"[Exploration] execute_pick on handle {handle_match} succeeded.")
                # 3. Try to pull the drawer (execute_pull)
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer,
                    handle=handle_match,
                    location=robot_pos
                )
                print(f"[Exploration] execute_pull on {drawer} using {handle_match} succeeded.")
            except Exception as e:
                print(f"[Exploration] Exception during pull attempt: {e}")
                # Exploration goal is to reveal if a predicate is missing: e.g., drawer-locked/unlocked
                # or holding/robot-free/etc. Based on the error, user can diagnose domain issues.
                # You might also log e to system or add more sophisticated checking here.

            # Optionally, try to push the drawer closed
            try:
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=drawer,
                    location=robot_pos
                )
                print(f"[Exploration] execute_push on {drawer} succeeded.")
            except Exception as e:
                print(f"[Exploration] Exception during push attempt: {e}")

        # --- Try pick/place on other objects (objects on floor) ---
        for obj in object_on_floor_names:
            obj_pos = positions[obj]
            # Move to object position
            try:
                if robot_initial_pos is not None:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_pos=robot_initial_pos,
                        to_pos=obj_pos
                    )
                    print(f"[Exploration] execute_go to {obj} ({obj_pos}) succeeded.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_go to {obj}: {e}")

            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target=obj,
                    location=obj_pos
                )
                print(f"[Exploration] execute_pick on {obj} succeeded.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick({obj}): {e}")

            # Optionally, try placing in one of the drawers
            for drawer in drawer_names:
                if drawer in positions:
                    drawer_pos = positions[drawer]
                    try:
                        obs, reward, done = execute_place(
                            env,
                            task,
                            target=obj,
                            drawer=drawer,
                            location=drawer_pos
                        )
                        print(f"[Exploration] execute_place {obj} in {drawer} succeeded.")
                    except Exception as e:
                        print(f"[Exploration] Exception during execute_place({obj},{drawer}): {e}")

        print("[Exploration] Exploration phase finished.")

        # === --- End of Exploration, Ready for Oracle Plan Execution --- ===
        print("[Task] You can now run the plan using only the available skills.")
        # Example Oracle Plan Executor (stub):
        # for action in oracle_plan:
        #     if action['name'] in ['execute_pick', 'execute_place', 'execute_pull', 'execute_push', 'execute_go']:
        #         try:
        #             # Map action name to skill function and pass required args
        #             # E.g. execute_pick(env, task, target, location)
        #             skill_fn = globals()[action['name']]
        #             obs, reward, done = skill_fn(
        #                 env, task,
        #                 **action['parameters']  # Parameters as required by each skill signature
        #             )
        #             print(f"[Task] {action['name']} executed with {action['parameters']}")
        #             if done:
        #                 print("[Task] Plan execution indicates done flag, terminating early.")
        #                 break
        #         except Exception as e:
        #             print(f"[Task] Exception executing {action['name']}: {e}")

        # TODO: (When deploying on real task): 
        #  - Replace hardcoded objects/locations above with those parsed from actual observation.
        #  - Parse and execute the plan step-by-step using only available skills.
        #  - After exploration, rerun planner/update domain as needed if missing predicates are found.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()