# run_skeleton_task.py (Completed for Exploration and Predicate Diagnosis)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine any skill functions!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    missing_predicate_found = False
    missing_predicate = None
    
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Logging initial environment
        print("[Info] Initial object positions:", positions)
        
        # --- EXPLORATION PHASE ---
        # The goal here is to identify the missing predicate (e.g., lock, temperature, durability, etc.)
        # We will use available skills to manipulate/interact and observe the effects.
        print("[Exploration] Begin missing predicate exploration...")

        # Step 1: Try known skills and observe anomalies/errors to infer missing predicate
        # Since we do not know the problem instance's content, perform systematic exploration
        
        # Let's iterate through all movable objects/locations available from positions
        obj_names = []
        loc_names = []
        drawer_names = []
        handle_names = []
        for name, pos in positions.items():
            # Simple heuristic for type detection
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'obj' in name or 'ball' in name or 'item' in name:
                obj_names.append(name)
            elif 'loc' in name or 'room' in name or 'table' in name:
                loc_names.append(name)
        
        # Fallback: Use all names as objects; user must implement this in object_positions
        if not obj_names and positions:
            obj_names = list(positions.keys())
        
        # Assume at least one location is known
        if not loc_names and positions:
            loc_names = list(positions.keys())
        
        # Safety: Only operate if objects and locations are found
        if not obj_names or not loc_names:
            print("[Exploration] Could not determine object or location names for exploration.")
            return
        
        exploration_results = {}
        
        # Try to execute each available skill and watch for errors/state issues
        for obj in obj_names:
            for loc in loc_names:
                # Try picking: does it work?
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(
                        env, task, obj, loc
                    )
                    exploration_results[(obj, 'execute_pick')] = (True, None)
                    print(f"   [Result] execute_pick succeeded on {obj} at {loc}")
                except Exception as e:
                    print(f"   [Error] execute_pick failed for {obj} at {loc}: {e}")
                    exploration_results[(obj, 'execute_pick')] = (False, str(e))
                
                # Try placing if pick succeeded
                if exploration_results.get((obj, 'execute_pick'), (False,))[0]:
                    # Use a drawer if available
                    if drawer_names:
                        for d in drawer_names:
                            try:
                                print(f"[Exploration] Trying execute_place on {obj} into drawer {d} at {loc}")
                                obs, reward, done = execute_place(
                                    env, task, obj, d, loc
                                )
                                exploration_results[(obj, 'execute_place')] = (True, None)
                                print(f"   [Result] execute_place succeeded for {obj} in {d}")
                            except Exception as e:
                                print(f"   [Error] execute_place failed for {obj} in {d} at {loc}: {e}")
                                exploration_results[(obj, 'execute_place')] = (False, str(e))
                            break
                    # Try to push/pull if possible
                    if drawer_names and handle_names:
                        for d in drawer_names:
                            for h in handle_names:
                                try:
                                    print(f"[Exploration] Trying execute_pull on drawer {d} handle {h} at {loc}")
                                    obs, reward, done = execute_pull(
                                        env, task, d, h, loc
                                    )
                                    exploration_results[(d, 'execute_pull')] = (True, None)
                                    print(f"   [Result] execute_pull succeeded for {d} with handle {h}")
                                except Exception as e:
                                    print(f"   [Error] execute_pull failed for {d} with {h} at {loc}: {e}")
                                    exploration_results[(d, 'execute_pull')] = (False, str(e))
                                break
                            break

        # Try simple move between locations
        if len(loc_names) > 1:
            try:
                print(f"[Exploration] Trying execute_go from {loc_names[0]} to {loc_names[1]}")
                obs, reward, done = execute_go(
                    env, task, loc_names[0], loc_names[1]
                )
                exploration_results[('robot', 'execute_go')] = (True, None)
                print(f"   [Result] execute_go succeeded from {loc_names[0]} to {loc_names[1]}")
            except Exception as e:
                print(f"   [Error] execute_go failed: {e}")
                exploration_results[('robot', 'execute_go')] = (False, str(e))
        
        # After exploration, try to deduce missing predicate from errors if any
        print("[Exploration] Results summary:")
        for key, (succ, err) in exploration_results.items():
            print(f" - {key}: {'Success' if succ else 'Fail'} | {err if err else 'Okay'}")
            # Heuristic: Look for hints in errors
            if err is not None and not missing_predicate_found:
                lower_err = err.lower()
                # Look for hints from errors (simulate error strings)
                for pred in ["identified", "temperature-known", "weight-known", "durability-known", "lock-known"]:
                    if pred.replace("-", "_") in lower_err or pred in lower_err:
                        missing_predicate_found = True
                        missing_predicate = pred
                        print(f"   [Diagnosis] Possible missing predicate: {pred}")
                        break
        
        if not missing_predicate_found:
            print("[Exploration] No explicit missing predicate found. Review error diagnostics above.")
        else:
            print(f"[Exploration] Missing predicate likely: {missing_predicate}")

        # Optionally: raise or report to user for further action
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
