# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is likely required for the plan to succeed.
        # We will use the available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # For this exploration, we will:
        # 1. Move the robot to each location.
        # 2. Attempt to interact with objects (pick, pull, etc.) using available skills.
        # 3. Observe if any action fails due to a missing predicate, and log the result.

        # Get all objects and locations from positions
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # If no explicit locations, try to infer from object positions
        if not location_names:
            # Use unique positions as locations
            unique_locs = set()
            for pos in positions.values():
                unique_locs.add(tuple(np.round(pos, 2)))
            location_names = [f"loc_{i}" for i in range(len(unique_locs))]

        # Try to move to each location and interact with objects
        exploration_results = {}
        for loc in location_names:
            print(f"[Exploration] Moving to location: {loc}")
            try:
                # Try to move to the location using execute_go
                obs, reward, done = execute_go(env, task, from_location=None, to_location=loc)
                print(f"[Exploration] Robot moved to {loc}")
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

            # Try to pick up each object at this location
            for obj in object_names:
                print(f"[Exploration] Attempting to pick up {obj} at {loc}")
                try:
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    print(f"[Exploration] Picked up {obj} at {loc}")
                    exploration_results[(obj, loc, 'pick')] = True
                except Exception as e:
                    print(f"[Exploration] Failed to pick up {obj} at {loc}: {e}")
                    exploration_results[(obj, loc, 'pick')] = False

            # Try to interact with drawers (pull/push)
            for drawer in drawer_names:
                # Try to pull the drawer using any handle
                for handle in handle_names:
                    print(f"[Exploration] Attempting to pull {drawer} with handle {handle} at {loc}")
                    try:
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] Pulled {drawer} with {handle} at {loc}")
                        exploration_results[(drawer, handle, loc, 'pull')] = True
                    except Exception as e:
                        print(f"[Exploration] Failed to pull {drawer} with {handle} at {loc}: {e}")
                        exploration_results[(drawer, handle, loc, 'pull')] = False

                # Try to push the drawer
                print(f"[Exploration] Attempting to push {drawer} at {loc}")
                try:
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    print(f"[Exploration] Pushed {drawer} at {loc}")
                    exploration_results[(drawer, loc, 'push')] = True
                except Exception as e:
                    print(f"[Exploration] Failed to push {drawer} at {loc}: {e}")
                    exploration_results[(drawer, loc, 'push')] = False

            # Try to place objects in drawers
            for obj in object_names:
                for drawer in drawer_names:
                    print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc}")
                    try:
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] Placed {obj} in {drawer} at {loc}")
                        exploration_results[(obj, drawer, loc, 'place')] = True
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} in {drawer} at {loc}: {e}")
                        exploration_results[(obj, drawer, loc, 'place')] = False

            # Try to sweep objects
            for obj in object_names:
                print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                try:
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    print(f"[Exploration] Swept {obj} at {loc}")
                    exploration_results[(obj, loc, 'sweep')] = True
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")
                    exploration_results[(obj, loc, 'sweep')] = False

            # Try to use gripper
            print(f"[Exploration] Attempting to use gripper at {loc}")
            try:
                obs, reward, done = execute_gripper(env, task)
                print(f"[Exploration] Used gripper at {loc}")
                exploration_results[(loc, 'gripper')] = True
            except Exception as e:
                print(f"[Exploration] Failed to use gripper at {loc}: {e}")
                exploration_results[(loc, 'gripper')] = False

        # After exploration, print summary of results
        print("\n===== Exploration Summary =====")
        for key, success in exploration_results.items():
            print(f"{key}: {'Success' if success else 'Failed'}")

        print("\n[Exploration] If certain actions consistently fail with similar error messages,")
        print("[Exploration] it may indicate a missing predicate (e.g., lock-known, identified, etc.) required in the domain.")

        # TODO: After identifying the missing predicate, update the domain/problem accordingly and rerun the plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()