# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be explored.
        # We will attempt to use available skills to explore the state of drawer1.

        # Assumptions:
        # - There is a drawer named 'drawer1' and possibly a handle object for it.
        # - The robot can move to the drawer's location and attempt to interact with it.
        # - The available skills are: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Step 1: Find the location of drawer1 and its handle (if any)
        drawer_name = 'drawer1'
        handle_name = None
        drawer_location = None

        # Try to find the handle and location for drawer1
        for obj_name, pos in positions.items():
            if 'handle' in obj_name and drawer_name in obj_name:
                handle_name = obj_name
            if drawer_name in obj_name:
                drawer_location = pos

        # Fallback: If drawer_location is not found, use a default or first location
        if drawer_location is None:
            for obj_name, pos in positions.items():
                if 'drawer' in obj_name:
                    drawer_location = pos
                    break

        # Step 2: Move robot to the drawer's location
        # We assume the robot's current location is available in the observation or positions
        robot_location = None
        for obj_name, pos in positions.items():
            if 'robot' in obj_name:
                robot_location = pos
                break

        # If robot location is not found, use a default
        if robot_location is None:
            robot_location = (0, 0, 0)

        # Use execute_go to move to the drawer's location
        try:
            print(f"[Exploration] Moving robot from {robot_location} to drawer at {drawer_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 3: Try to pull the drawer to check its state (lock-known or closed)
        # If handle is available, try to pick it and pull
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick handle: {handle_name}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=handle_name,
                    location=drawer_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_pick: {e}")

            try:
                print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_name=handle_name,
                    location=drawer_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_pull: {e}")
        else:
            print("[Exploration] No handle found for drawer, skipping pull.")

        # Step 4: Try to push the drawer (if open) to check for (drawer-closed drawer1)
        try:
            print(f"[Exploration] Attempting to push (close) drawer: {drawer_name}")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Error during execute_push: {e}")

        # Step 5: Record the result and check for the missing predicate
        # (In a real system, you would now check the state for (drawer-closed drawer1))
        print("[Exploration] Exploration phase complete. Check logs or state for (drawer-closed drawer1).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()