# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from PDDL and feedback) ===
        # Objects
        objects = ['dice1', 'dice2', 'handle1', 'handle2', 'handle3', 'trash']
        # Drawers
        drawers = ['drawer1', 'drawer2', 'drawer3']
        # Locations
        locations = ['floor', 'drawer-area', 'ready-pose']

        # === Validate Object Existence (per feedback) ===
        for obj in objects:
            if obj not in positions:
                print(f"[Warning] Object '{obj}' not found in environment positions. Skipping.")
        # Remove missing objects from the list
        objects = [obj for obj in objects if obj in positions]

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore the environment to find out which predicate is missing.
        # For this, we can try to interact with all objects and drawers, and check for errors or missing preconditions.
        # We use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Move robot to each location to ensure all locations are reachable
        current_location = 'ready-pose'
        for loc in locations:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Moving robot from {current_location} to {loc}")
                obs, reward, done = execute_go(env, task, current_location, loc)
                current_location = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # 2. Try to pick up each object from the floor
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to pick up {obj} at 'floor'")
                obs, reward, done = execute_pick(env, task, obj, 'floor')
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}: {e}")

        # 3. Try to open each drawer using its handle (if handle is present)
        handle_map = {'drawer1': 'handle1', 'drawer2': 'handle2', 'drawer3': 'handle3'}
        for drawer in drawers:
            handle = handle_map.get(drawer)
            if handle and handle in objects:
                try:
                    # Move to drawer-area if not already there
                    if current_location != 'drawer-area':
                        print(f"[Exploration] Moving to drawer-area for {drawer}")
                        obs, reward, done = execute_go(env, task, current_location, 'drawer-area')
                        current_location = 'drawer-area'
                    # Pick the handle
                    print(f"[Exploration] Picking handle {handle} for {drawer}")
                    obs, reward, done = execute_pick(env, task, handle, 'floor')
                    # Pull the drawer open
                    print(f"[Exploration] Pulling {drawer} using {handle}")
                    obs, reward, done = execute_pull(env, task, drawer, handle, 'drawer-area')
                except Exception as e:
                    print(f"[Exploration] Failed to open {drawer} with {handle}: {e}")

        # 4. Try to place dice into each drawer (if open and not full)
        for drawer in drawers:
            for obj in ['dice1', 'dice2']:
                if obj in objects:
                    try:
                        print(f"[Exploration] Attempting to place {obj} into {drawer}")
                        obs, reward, done = execute_place(env, task, obj, drawer, 'drawer-area')
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} into {drawer}: {e}")

        # 5. Try to push drawers closed
        for drawer in drawers:
            try:
                print(f"[Exploration] Attempting to push (close) {drawer}")
                obs, reward, done = execute_push(env, task, drawer, 'drawer-area')
            except Exception as e:
                print(f"[Exploration] Failed to push {drawer}: {e}")

        # 6. Try to sweep any object on the floor
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to sweep {obj} at 'floor'")
                obs, reward, done = execute_sweep(env, task, obj, 'floor')
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 7. Try to use gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        # === End of Exploration ===
        print("[Exploration] Completed. Review logs for missing predicate or action failures.")

        # === Main Task Plan (Example) ===
        # After exploration, you would implement the actual oracle plan here using the available skills.
        # For this code, we focus on the exploration phase as per the instructions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
