# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available exploration skills to probe the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to check for missing predicates.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we will try to use each skill in a safe way and observe the environment's response.
        # This will help us infer which predicate is missing or required for the plan to succeed.

        # Example: Try to pick up each object on the floor, try to open/close drawers, etc.

        print("[Exploration] Starting exploration to identify missing predicates...")

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']

        # Try to pick up each object on the floor
        for obj in object_names:
            try:
                obj_info = positions[obj]
                if obj_info.get('on_floor', False):
                    loc = obj_info.get('location', None)
                    if loc is not None:
                        print(f"[Exploration] Trying to pick up {obj} at {loc}...")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Successfully picked up {obj}.")
                        else:
                            print(f"[Exploration] Could not pick up {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during pick: {e}")

        # Try to open each drawer using its handle
        for drawer in drawer_names:
            try:
                drawer_info = positions[drawer]
                handle = drawer_info.get('handle', None)
                loc = drawer_info.get('location', None)
                if handle is not None and loc is not None:
                    print(f"[Exploration] Trying to pull (open) drawer {drawer} with handle {handle} at {loc}...")
                    # First, pick the handle
                    try:
                        obs, reward, done = execute_pick(env, task, handle, loc)
                    except Exception as e:
                        print(f"[Exploration] Exception during handle pick: {e}")
                    # Then, try to pull the drawer
                    try:
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] Successfully pulled (opened) drawer {drawer}.")
                        else:
                            print(f"[Exploration] Could not pull (open) drawer {drawer}.")
                    except Exception as e:
                        print(f"[Exploration] Exception during drawer pull: {e}")
            except Exception as e:
                print(f"[Exploration] Exception during drawer exploration: {e}")

        # Try to push (close) each open drawer
        for drawer in drawer_names:
            try:
                drawer_info = positions[drawer]
                loc = drawer_info.get('location', None)
                if loc is not None:
                    print(f"[Exploration] Trying to push (close) drawer {drawer} at {loc}...")
                    try:
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        if done:
                            print(f"[Exploration] Successfully pushed (closed) drawer {drawer}.")
                        else:
                            print(f"[Exploration] Could not push (close) drawer {drawer}.")
                    except Exception as e:
                        print(f"[Exploration] Exception during drawer push: {e}")
            except Exception as e:
                print(f"[Exploration] Exception during drawer push exploration: {e}")

        # Try to move between locations
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            print(f"[Exploration] Trying to move from {from_loc} to {to_loc}...")
            try:
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
                else:
                    print(f"[Exploration] Could not move from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] Exception during move: {e}")

        # Try to sweep each object
        for obj in object_names:
            try:
                obj_info = positions[obj]
                loc = obj_info.get('location', None)
                if loc is not None:
                    print(f"[Exploration] Trying to sweep {obj} at {loc}...")
                    try:
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Successfully swept {obj}.")
                        else:
                            print(f"[Exploration] Could not sweep {obj}.")
                    except Exception as e:
                        print(f"[Exploration] Exception during sweep: {e}")
            except Exception as e:
                print(f"[Exploration] Exception during sweep exploration: {e}")

        # Try to use gripper (if applicable)
        try:
            print("[Exploration] Trying to use gripper...")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Successfully used gripper.")
            else:
                print("[Exploration] Could not use gripper.")
        except Exception as e:
            print(f"[Exploration] Exception during gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration ===

        # === (Optional) Main Task Plan Execution ===
        # At this point, after exploration, you would normally execute the oracle plan.
        # For this code, the focus is on exploration to identify missing predicates as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()